/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ParserXmlUnitTest.cpp
 *
 *  Created on: Mar 24, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserXmlUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Atom/atom.hpp"
#include "Atom/AtomObserver.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/XmlParser.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserXmlUnitTest );

static string waterXml = "\
<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n\
<!DOCTYPE scafacos_test SYSTEM 'scafacos_test.dtd'>\n\
\t<scafacos_test name=\"cloud-wall\" description=\"A 3D-periodic system of two walls charges and a random cloud of charges.\" reference_method=\"ewald\" error_potential=\"1.0e-13\" error_field=\"1.0e-13\">\n\
\t\t<configuration offset=\"0 0 0\" box_a=\"10 0 0\" box_b=\"0 10 0\" box_c=\"0 0 10\" periodicity=\"1 1 1\" epsilon=\"metallic\">\n\
\t\t\t<particle position=\"0 0 4.5\" q=\"1\" potential=\"1.318858869367693\" field=\"0.27973426575304 -0.1382533033408908 5.200985873286221\"/>\n\
\t\t\t<particle position=\"0 0 5.5\" q=\"1\" potential=\"1.318858869367693\" field=\"0.27973426575304 -0.1382533033408908 5.200985873286221\"/>\n\
\t\t\t<particle position=\"0 0 6.5\" q=\"1\" potential=\"1.318858869367693\" field=\"0.27973426575304 -0.1382533033408908 5.200985873286221\"/>\n\
\t\t</configuration>\n\
\t</scafacos_test>\n";
static string waterMultiXml = "\n";

void ParserXmlUnitTest::setUp() {
  World::getInstance();

  parser = new FormatParser<xml>();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserXmlUnitTest::tearDown()
{
  delete parser;
  ChangeTracker::purgeInstance();
  World::purgeInstance();
  AtomObserver::purgeInstance();
}

/************************************ tests ***********************************/

void ParserXmlUnitTest::readwriteXmlTest() {
  cout << "Testing the XML parser." << endl;
  stringstream input;
  input << waterXml;
  parser->load(&input);
  input.clear();

  CPPUNIT_ASSERT_EQUAL(3, const_cast<const World &>(World::getInstance()).numAtoms());

  // store and parse in again
  {
//    std::string first;
//    std::string second;
    std::stringstream output;
    std::vector<const atom *> atoms = const_cast<const World &>(World::getInstance()).
        getAllAtoms();
    parser->save(&output, atoms);
    std::cout << output.str();

    {
      delete parser;
      parser = new FormatParser<xml>();
      std::stringstream input(waterXml);
      parser->load(&input);
    }

    FormatParser<xml> *parser_control = new FormatParser<xml>();
    parser_control->load(&output);
    CPPUNIT_ASSERT( parser->data == parser_control->data );
    delete parser_control;
  }
}
