/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ParserPdbUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserPdbUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Atom/atom.hpp"
#include "Atom/AtomObserver.hpp"
#include "CodePatterns/Log.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/PdbParser.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserPdbUnitTest );

//----|----*|---||--*||---|***|-------|-------|-------|-----|---------------|-|-
//000000011111111112222222222333333333344444444445555555555666666666677777777778
//345678901234567890123456789012345678901234567890123456789012345678901234567890
static string waterPdb = "\
REMARK  This is a test water molecule as written by TREMOLO.\n\
ATOM      1  OT  GMT-    0       1.583   1.785   1.480  1.00178.02           O-2\n\
ATOM      2  HT  GMT-    0       1.186   1.643   2.213  1.00103.58           H+1\n\
ATOM      3  HT  GMT-    0       2.642   1.896   1.730  1.00126.00           H+1\n\
ATOM      4  OT  GMT-    1       3.583   1.785   1.480  1.00178.02           O-2\n\
ATOM      5  HT  GMT-    1       3.186   1.643   2.213  1.00103.58           H+1\n\
ATOM      6  HT  GMT-    1       4.642   1.896   1.730  1.00126.00           H+1\n\
CONECT    1    2    3\n\
CONECT    2    1\n\
CONECT    3    1\n\
CONECT    4    5    6\n\
CONECT    5    4\n\
CONECT    6    4\n\
END";

void ParserPdbUnitTest::setUp() {
  World::getInstance();

  parser = new FormatParser<pdb>();

  setVerbosity(3);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserPdbUnitTest::tearDown() 
{
  delete parser;
  ChangeTracker::purgeInstance();
  World::purgeInstance();
  AtomObserver::purgeInstance();
}

/************************************ tests ***********************************/

void ParserPdbUnitTest::readwritePdbTest() {
  stringstream input;
  input << waterPdb;
  parser->load(&input);
  input.clear();

  CPPUNIT_ASSERT_EQUAL(6, const_cast<const World &>(World::getInstance()).numAtoms());

  stringstream output;
  std::vector<const atom *> atoms = const_cast<const World &>(World::getInstance()).
      getAllAtoms();
  parser->save(&output, atoms);

  //  std::cout << "Save PDB is:" << std::endl;
//  std::cout << output.str() << std::endl;

  input << output.str();
  FormatParser<pdb>* parser2 = new FormatParser<pdb>();
  parser2->load(&input);

  CPPUNIT_ASSERT_EQUAL(12, World::getInstance().numAtoms());

  delete parser2;
}


void ParserPdbUnitTest::printCoordinateTest() 
{
  CPPUNIT_ASSERT_EQUAL( 
      std::string("1234.678"), FormatParser<pdb>::printCoordinate(1234.678) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("1.234"), FormatParser<pdb>::printCoordinate(1.234) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("0.001"), FormatParser<pdb>::printCoordinate(0.001) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("0.100"), FormatParser<pdb>::printCoordinate(0.100) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("1234567."), FormatParser<pdb>::printCoordinate(1234567.) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("123456.7"), FormatParser<pdb>::printCoordinate(123456.7) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("-1234.56"), FormatParser<pdb>::printCoordinate(-1234.56) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("4.990"), FormatParser<pdb>::printCoordinate(4.99) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("4.999"), FormatParser<pdb>::printCoordinate(4.999) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("5.000"), FormatParser<pdb>::printCoordinate(4.9999) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("5.000"), FormatParser<pdb>::printCoordinate(4.99999) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("5.001"), FormatParser<pdb>::printCoordinate(5.001) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("5.000"), FormatParser<pdb>::printCoordinate(5.0001) );
  CPPUNIT_ASSERT_EQUAL( 
      std::string("5.000"), FormatParser<pdb>::printCoordinate(5.00001) );
}

