/*
 * XmlParser.hpp
 *
 *  Created on: Mar 23, 2012
 *      Author: heber
 */

#ifndef XMLPARSER_HPP_
#define XMLPARSER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <string>

#include "FormatParser.hpp"
#include "FormatParserTrait.hpp"
#include "FormatParserInterface.hpp"
#include "FormatParser_common.hpp"
#include "ParserTypes.hpp"

#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"

// declaration of specialized FormatParserTrait
template<>
struct FormatParserTrait<xml>
{
  //!> Name of the parser
  static const std::string name;
  //!> suffix of the files the parser understands to read and write
  static const std::string suffix;
  //!> ParserTypes enumeration for the parser
  static const enum ParserTypes type;
};

class ParserXmlUnitTest;

/**
 * Parser for XYZ files.
 */
template <>
class FormatParser< xml >  : virtual public FormatParserInterface, public FormatParser_common
{
  //!> grant unit test access to private parts
  friend class ParserXmlUnitTest;
  //!> grant mem debugger access
  friend void AddStaticEntitiestoIgnoreList();
public:
  FormatParser();
  virtual ~FormatParser();
  void load(std::istream* file);
  void save(std::ostream* file, const std::vector<const atom *> &atoms);

protected:
  void AtomInserted(atomId_t);
  void AtomRemoved(atomId_t);

private:
  //!> structure that contains all information from the xml file
  struct scafacos {
    std::string name;
    std::string description;
    std::string reference_method;
    double error_potential;
    double error_field;
    struct configuration {
      Vector offset;
      RealSpaceMatrix box;
      bool periodicity[NDIM];
      std::string epsilon;

      struct particle {
        Vector position;
        double q;
        double potential;
        Vector field;

        bool operator==(const particle &p) const;
        bool operator!=(const particle &p) const {
          return !((*this) == p);
        }
      };
      std::vector<struct particle> p;

      bool operator==(const configuration &c) const;
      bool operator!=(const configuration &c) const {
        return !((*this) == c);
      }
    } config;

    bool operator==(const scafacos &s) const;
    bool operator!=(const scafacos &s) const {
      return !((*this) == s);
    }
  } data;

  //!> additional parser-specific information for an atom.
  struct additionalAtomInfo {
    /** Default constructor for additionalAtomInfo.
     *
     * Sets all parser-specific values to zero.
     */
    additionalAtomInfo() :
      charge(0.),
      potential(0),
      field(zeroVec)
    {}

    /** Default constructor for additionalAtomInfo.
     *
     */
    additionalAtomInfo(double _charge, double _potential, const Vector& _field) :
      charge(_charge),
      potential(_potential),
      field(_field)
    {}

    //!> charge of the atom
    double charge;
    //!> potential at position of atom
    double potential;
    //!> field at position of atom
    Vector field;
  };

  //!> typedef for map to associate additional parser-specific information to each atom.
  typedef std::map< atomId_t, additionalAtomInfo> AtomInfoMap_t;

  //!> map to associate additional parser-specific information to each atom.
  AtomInfoMap_t additionalAtomData;

  //!> static instance with default additional atom information
  static additionalAtomInfo defaultAtomInfo;

  /** Getter for additionalAtomInfo.
   *
   * @param _atom constant ref to atom
   * @return constant reference to additional atom info container,
   *         otherwise to default atom info
   */
  const additionalAtomInfo& getAtomData(const atom &_atom) const;
};


#endif /* XMLPARSER_HPP_ */
