/** \file FormatParserStorage.hpp
 *
 *  date: Jun, 22 2010
 *  author: heber
 *
 */

#ifndef FORMATPARSERSTORAGE_HPP_
#define FORMATPARSERSTORAGE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Singleton.hpp"

#include <boost/filesystem/path.hpp>
#include <iosfwd>
#include <map>
#include <string>
#include <vector>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "FormatParser.hpp"
#include "ParserTypes.hpp"

class atom;
class FormatParserInterface;

class FormatParserStorage : public Singleton<FormatParserStorage> {
  friend class Singleton<FormatParserStorage>;
public:

  bool add(std::string type);
  bool add(ParserTypes type);

  bool load(boost::filesystem::path filename);
  bool load(std::string &filename);
  bool load(std::istream &input, std::string &suffix);
  bool save(std::ostream &output, std::string suffix, const std::vector<const atom *> &atoms);
  bool saveSelectedAtoms(std::ostream &output, std::string suffix);
  bool saveSelectedMolecules(std::ostream &output, std::string suffix);
  bool saveWorld(std::ostream &output, std::string suffix);

  void setOutputFormat(ParserTypes type);
  void setOutputFormat(std::string type);
  
  FormatParserInterface &get(ParserTypes _type);

  ParserTypes getTypeFromName(std::string type);
  ParserTypes getTypeFromSuffix(std::string type);

  const std::string &getSuffixFromType(ParserTypes type);
  const std::string &getNameFromType(ParserTypes type);

  void SetOutputPrefixForAll(std::string &_prefix);
  bool isAbleToSave();
  void SaveAll();

private:
  // private constructors as this is a singleton
  FormatParserStorage();
  ~FormatParserStorage();
  
  // list of allocated parsers
  std::vector<FormatParserInterface *> ParserList;

  // list of allocated strams
  std::vector<std::ofstream *> ParserStream;

  // which parser is already present
  std::vector<bool> ParserPresent;

  // which parser is already present
  std::vector<bool> ParserDesiredOutputFormat;

  // default suffix of each parser type
  std::map<ParserTypes, std::string> ParserSuffixes;
  std::map<std::string, ParserTypes> ParserLookupSuffixes;

  // function pointers to each add...()
  std::map< ParserTypes, void (FormatParserStorage::*)() > ParserAddFunction;

  // type name of each parser type and reverse lookup
  std::map<ParserTypes, std::string> ParserNames;
  std::map<std::string, ParserTypes> ParserLookupNames;

  // prefix of the filenames to use on save
  std::string prefix;

  static const std::string unknownTypeString;

public:
  template<ParserTypes Ptype> void addParser()
  {
    if (!ParserPresent[Ptype]) {
      ParserList[Ptype] = new FormatParser<Ptype>();
      ParserPresent[Ptype] = true;
    } else {
      ASSERT(ParserNames.find(Ptype) != ParserNames.end(),
          "FormatParserStorage::addParser() - ParserNames unknown for type"+toString((size_t)Ptype)+".");
      ASSERT(ParserSuffixes.find(Ptype) != ParserSuffixes.end(),
          "FormatParserStorage::addParser() - ParserSuffixes unknown for type"+toString((size_t)Ptype)+".");
      LOG(2, "INFO: Parser " << ParserNames[Ptype] << " is already present." << std::endl
          << "Note that you don't need to add '-o " 
          << ParserSuffixes[Ptype] << "' if the input file is non-empty and of type " 
          << ParserSuffixes[Ptype] << "." << std::endl);
    }
  }

  template<ParserTypes Ptype> FormatParser<Ptype> &getParser() 
  {
    if(!ParserPresent[Ptype])
      addParser< Ptype >();
    return dynamic_cast<FormatParser<Ptype> &>(*ParserList[Ptype]);
  }
};

#endif // FORMATPARSERSTORAGE_HPP_
