/*
 * SurfaceCorrelationAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Actions/AnalysisAction/SurfaceCorrelationAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "analysis_correlation.hpp"
#include "boundary.hpp"
#include "linkedcell.hpp"
#include "Helpers/Verbose.hpp"
#include "Helpers/Log.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "tesselation.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

const char AnalysisSurfaceCorrelationAction::NAME[] = "surface-correlation";

AnalysisSurfaceCorrelationAction::AnalysisSurfaceCorrelationAction() :
  Action(NAME)
{}

AnalysisSurfaceCorrelationAction::~AnalysisSurfaceCorrelationAction()
{}

void AnalysisSurfaceCorrelation(std::vector< element *> &elements, molecule *mol, double BinStart, double BinWidth, double BinEnd, std::string &outputname, std::string &binoutputname, bool periodic) {
  ValueStorage::getInstance().setCurrentValue("elements", elements);
  ValueStorage::getInstance().setCurrentValue("molecule-by-id", mol);
  ValueStorage::getInstance().setCurrentValue("bin-start", BinStart);
  ValueStorage::getInstance().setCurrentValue("bin-width", BinWidth);
  ValueStorage::getInstance().setCurrentValue("bin-end", BinEnd);
  ValueStorage::getInstance().setCurrentValue("output-file", outputname);
  ValueStorage::getInstance().setCurrentValue("bin-output-file", binoutputname);
  ValueStorage::getInstance().setCurrentValue("periodic", periodic);
  ActionRegistry::getInstance().getActionByName(AnalysisSurfaceCorrelationAction::NAME)->call(Action::NonInteractive);
};


Dialog* AnalysisSurfaceCorrelationAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryMolecule("molecule-by-id", ValueStorage::getInstance().getDescription("molecule-by-id"));
  dialog->queryElements("elements", ValueStorage::getInstance().getDescription("elements"));
  dialog->queryDouble("bin-start", ValueStorage::getInstance().getDescription("bin-start"));
  dialog->queryDouble("bin-width", ValueStorage::getInstance().getDescription("bin-width"));
  dialog->queryDouble("bin-end", ValueStorage::getInstance().getDescription("bin-end"));
  dialog->queryString("output-file", ValueStorage::getInstance().getDescription("output-file"));
  dialog->queryString("bin-output-file", ValueStorage::getInstance().getDescription("bin-output-file"));
  dialog->queryBoolean("periodic", ValueStorage::getInstance().getDescription("periodic"));

  return dialog;
}

Action::state_ptr AnalysisSurfaceCorrelationAction::performCall() {
  int ranges[3] = {1, 1, 1};
  double BinEnd = 0.;
  double BinStart = 0.;
  double BinWidth = 0.;
  molecule *Boundary = NULL;
  string outputname;
  string binoutputname;
  bool periodic;
  ofstream output;
  ofstream binoutput;
  std::vector<const element *> elements;
  string type;
  Vector Point;
  BinPairMap *binmap = NULL;

  // obtain information
  ValueStorage::getInstance().queryCurrentValue("molecule-by-id", Boundary);
  ValueStorage::getInstance().queryCurrentValue("elements", elements);
  ValueStorage::getInstance().queryCurrentValue("bin-start", BinStart);
  ValueStorage::getInstance().queryCurrentValue("bin-width", BinWidth);
  ValueStorage::getInstance().queryCurrentValue("bin-end", BinEnd);
  ValueStorage::getInstance().queryCurrentValue("output-file", outputname);
  ValueStorage::getInstance().queryCurrentValue("bin-output-file", binoutputname);
  ValueStorage::getInstance().queryCurrentValue("periodic", periodic);

  // execute action
  output.open(outputname.c_str());
  binoutput.open(binoutputname.c_str());
  ASSERT(Boundary != NULL, "No molecule specified for SurfaceCorrelation.");
  const double radius = 4.;
  double LCWidth = 20.;
  if (BinEnd > 0) {
    if (BinEnd > 2.*radius)
      LCWidth = BinEnd;
    else
      LCWidth = 2.*radius;
  }

  // get the boundary
  class Tesselation *TesselStruct = NULL;
  const LinkedCell *LCList = NULL;
  // find biggest molecule
  std::vector<molecule*> molecules = World::getInstance().getSelectedMolecules();
  std::cout << "There are " << molecules.size() << " selected molecules." << std::endl;
  LCList = new LinkedCell(Boundary, LCWidth);
  FindNonConvexBorder(Boundary, TesselStruct, LCList, radius, NULL);
  CorrelationToSurfaceMap *surfacemap = NULL;
  if (periodic)
    surfacemap = PeriodicCorrelationToSurface( molecules, elements, TesselStruct, LCList, ranges);
  else
    surfacemap = CorrelationToSurface( molecules, elements, TesselStruct, LCList);
  delete LCList;
  OutputCorrelationToSurface(&output, surfacemap);
  // check whether radius was appropriate
  {
    double start; double end;
    GetMinMax( surfacemap, start, end);
    if (LCWidth < end)
      DoeLog(1) && (eLog()<< Verbose(1) << "Linked Cell width is smaller than the found range of values! Bins can only be correct up to: " << radius << "." << endl);
  }
  binmap = BinData( surfacemap, BinWidth, BinStart, BinEnd );
  OutputCorrelation ( &binoutput, binmap );
  delete TesselStruct;  // surfacemap contains refs to triangles! delete here, not earlier!
  delete(binmap);
  delete(surfacemap);
  output.close();
  binoutput.close();
  return Action::success;
}

Action::state_ptr AnalysisSurfaceCorrelationAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr AnalysisSurfaceCorrelationAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool AnalysisSurfaceCorrelationAction::canUndo() {
  return true;
}

bool AnalysisSurfaceCorrelationAction::shouldUndo() {
  return true;
}

const string AnalysisSurfaceCorrelationAction::getName() {
  return NAME;
}
