/*
 * World.cpp
 *
 *  Created on: Feb 3, 2010
 *      Author: crueger
 */

#include "Helpers/MemDebug.hpp"

#include "World.hpp"
#include "AtomSet_impl.hpp"

#include "atom.hpp"
#include "config.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "ThermoStatContainer.hpp"
#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomDescriptor_impl.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"
#include "Descriptors/MoleculeDescriptor_impl.hpp"
#include "Descriptors/SelectiveIterator_impl.hpp"
#include "Actions/ManipulateAtomsProcess.hpp"
#include "Helpers/Assert.hpp"
#include "Box.hpp"
#include "Matrix.hpp"

#include "Patterns/Singleton_impl.hpp"

using namespace std;

/******************************* getter and setter ************************/
periodentafel *&World::getPeriode(){
  return periode;
}

config *&World::getConfig(){
  return configuration;
}

// Atoms

atom* World::getAtom(AtomDescriptor descriptor){
  return descriptor.find();
}

vector<atom*> World::getAllAtoms(AtomDescriptor descriptor){
  return descriptor.findAll();
}

vector<atom*> World::getAllAtoms(){
  return getAllAtoms(AllAtoms());
}

int World::numAtoms(){
  return atoms.size();
}

// Molecules

molecule *World::getMolecule(MoleculeDescriptor descriptor){
  return descriptor.find();
}

std::vector<molecule*> World::getAllMolecules(MoleculeDescriptor descriptor){
  return descriptor.findAll();
}

std::vector<molecule*> World::getAllMolecules(){
  return getAllMolecules(AllMolecules());
}

int World::numMolecules(){
  return molecules_deprecated->ListOfMolecules.size();
}

// system

Box& World::getDomain() {
  return *cell_size;
}

void World::setDomain(const Matrix &mat){
  *cell_size = mat;
}

void World::setDomain(double * matrix)
{
  Matrix M = ReturnFullMatrixforSymmetric(matrix);
  cell_size->setM(M);
}

std::string World::getDefaultName() {
  return defaultName;
}

void World::setDefaultName(std::string name)
{
  defaultName = name;
};

class ThermoStatContainer * World::getThermostats()
{
  return Thermostats;
}


int World::getExitFlag() {
  return ExitFlag;
}

void World::setExitFlag(int flag) {
  if (ExitFlag < flag)
    ExitFlag = flag;
}

/******************** Methods to change World state *********************/

molecule* World::createMolecule(){
  OBSERVE;
  molecule *mol = NULL;
  mol = NewMolecule();
  ASSERT(!molecules.count(currMoleculeId),"currMoleculeId did not specify an unused ID");
  mol->setId(currMoleculeId++);
  // store the molecule by ID
  molecules[mol->getId()] = mol;
  mol->signOn(this);
  return mol;
}

void World::destroyMolecule(molecule* mol){
  OBSERVE;
  destroyMolecule(mol->getId());
}

void World::destroyMolecule(moleculeId_t id){
  OBSERVE;
  molecule *mol = molecules[id];
  ASSERT(mol,"Molecule id that was meant to be destroyed did not exist");
  DeleteMolecule(mol);
  molecules.erase(id);
}

atom *World::createAtom(){
  OBSERVE;
  atomId_t id = getNextAtomId();
  atom *res = NewAtom(id);
  res->setWorld(this);
  // store the atom by ID
  atoms[res->getId()] = res;
  return res;
}


int World::registerAtom(atom *atom){
  OBSERVE;
  atomId_t id = getNextAtomId();
  atom->setId(id);
  atom->setWorld(this);
  atoms[atom->getId()] = atom;
  return atom->getId();
}

void World::destroyAtom(atom* atom){
  OBSERVE;
  int id = atom->getId();
  destroyAtom(id);
}

void World::destroyAtom(atomId_t id) {
  OBSERVE;
  atom *atom = atoms[id];
  ASSERT(atom,"Atom ID that was meant to be destroyed did not exist");
  DeleteAtom(atom);
  atoms.erase(id);
  releaseAtomId(id);
}

bool World::changeAtomId(atomId_t oldId, atomId_t newId, atom* target){
  OBSERVE;
  // in case this call did not originate from inside the atom, we redirect it,
  // to also let it know that it has changed
  if(!target){
    target = atoms[oldId];
    ASSERT(target,"Atom with that ID not found");
    return target->changeId(newId);
  }
  else{
    if(reserveAtomId(newId)){
      atoms.erase(oldId);
      atoms.insert(pair<atomId_t,atom*>(newId,target));
      return true;
    }
    else{
      return false;
    }
  }
}

ManipulateAtomsProcess* World::manipulateAtoms(boost::function<void(atom*)> op,std::string name,AtomDescriptor descr){
  return new ManipulateAtomsProcess(op, descr,name,true);
}

ManipulateAtomsProcess* World::manipulateAtoms(boost::function<void(atom*)> op,std::string name){
  return manipulateAtoms(op,name,AllAtoms());
}

/********************* Internal Change methods for double Callback and Observer mechanism ********/

void World::doManipulate(ManipulateAtomsProcess *proc){
  proc->signOn(this);
  {
    OBSERVE;
    proc->doManipulate(this);
  }
  proc->signOff(this);
}
/******************************* IDManagement *****************************/

// Atoms

atomId_t World::getNextAtomId(){
  // see if we can reuse some Id
  if(atomIdPool.empty()){
    return currAtomId++;
  }
  else{
    // we give out the first ID from the pool
    atomId_t id = *(atomIdPool.begin());
    atomIdPool.erase(id);
    return id;
  }
}

void World::releaseAtomId(atomId_t id){
  atomIdPool.insert(id);
  // defragmentation of the pool
  set<atomId_t>::reverse_iterator iter;
  // go through all Ids in the pool that lie immediately below the border
  while(!atomIdPool.empty() && *(atomIdPool.rbegin())==(currAtomId-1)){
    atomIdPool.erase(--currAtomId);
  }
}

bool World::reserveAtomId(atomId_t id){
  if(id>=currAtomId ){
    // add all ids between the new one and current border as available
    for(atomId_t pos=currAtomId; pos<id; ++pos){
      atomIdPool.insert(pos);
    }
    currAtomId=id+1;
    return true;
  }
  else if(atomIdPool.count(id)){
    atomIdPool.erase(id);
    return true;
  }
  else{
    // this ID could not be reserved
    return false;
  }
}

// Molecules

/******************************* Iterators ********************************/

// external parts with observers

CONSTRUCT_SELECTIVE_ITERATOR(atom*,World::AtomSet,AtomDescriptor);

World::AtomIterator
World::getAtomIter(AtomDescriptor descr){
    return AtomIterator(descr,atoms);
}

World::AtomIterator
World::getAtomIter(){
    return AtomIterator(AllAtoms(),atoms);
}

World::AtomIterator
World::atomEnd(){
  return AtomIterator(AllAtoms(),atoms,atoms.end());
}

// Internal parts, without observers

// Build the AtomIterator from template
CONSTRUCT_SELECTIVE_ITERATOR(atom*,World::AtomSet::set_t,AtomDescriptor);


World::internal_AtomIterator
World::getAtomIter_internal(AtomDescriptor descr){
  return internal_AtomIterator(descr,atoms.getContent());
}

World::internal_AtomIterator
World::atomEnd_internal(){
  return internal_AtomIterator(AllAtoms(),atoms.getContent(),atoms.end_internal());
}

// build the MoleculeIterator from template
CONSTRUCT_SELECTIVE_ITERATOR(molecule*,World::MoleculeSet,MoleculeDescriptor);

World::MoleculeIterator World::getMoleculeIter(MoleculeDescriptor descr){
  return MoleculeIterator(descr,molecules);
}

World::MoleculeIterator World::moleculeEnd(){
  return MoleculeIterator(AllMolecules(),molecules,molecules.end());
}

/******************************* Singleton Stuff **************************/

World::World() :
    Observable("World"),
    periode(new periodentafel),
    configuration(new config),
    Thermostats(new ThermoStatContainer),
    ExitFlag(0),
    atoms(this),
    currAtomId(0),
    molecules(),
    currMoleculeId(0),
    molecules_deprecated(new MoleculeListClass(this))
{
  cell_size = new Box;
  Matrix domain;
  domain.at(0,0) = 20;
  domain.at(1,1) = 20;
  domain.at(2,2) = 20;
  cell_size->setM(domain);
  defaultName = "none";
  molecules_deprecated->signOn(this);
}

World::~World()
{
  molecules_deprecated->signOff(this);
  delete cell_size;
  delete molecules_deprecated;
  delete periode;
  delete configuration;
  delete Thermostats;
  MoleculeSet::iterator molIter;
  for(molIter=molecules.begin();molIter!=molecules.end();++molIter){
    DeleteMolecule((*molIter).second);
  }
  molecules.clear();
  AtomSet::iterator atIter;
  for(atIter=atoms.begin();atIter!=atoms.end();++atIter){
    DeleteAtom((*atIter).second);
  }
  atoms.clear();
}

// Explicit instantiation of the singleton mechanism at this point

CONSTRUCT_SINGLETON(World)

/******************************* deprecated Legacy Stuff ***********************/

MoleculeListClass *&World::getMolecules() {
  return molecules_deprecated;
}
