/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Woodcock.cpp
 *
 *  Created on: Aug 20, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Woodcock.hpp"

#include "CodePatterns/Log.hpp"
#include "Element/element.hpp"
#include "Helpers/defs.hpp"
#include "Parser/PcpParser_helper.hpp"
#include "Thermostats/ThermoStatContainer.hpp"

#include <sstream>

Woodcock::Woodcock(int _ScaleTempStep) :
  ScaleTempStep(_ScaleTempStep)
{}

Woodcock::Woodcock() :
  ScaleTempStep(25)
{}

Woodcock::~Woodcock()
{}

const char *ThermostatTraits<Woodcock>::name = "Woodcock";

std::string ThermostatTraits<Woodcock>::getName(){
  return ThermostatTraits<Woodcock>::name;
}

Thermostat *ThermostatTraits<Woodcock>::make(class ConfigFileBuffer * const fb){
  int ScaleTempStep;
  const int verbose = 0;
  ParseForParameter(verbose,fb,"Thermostat", 0, 2, 1, int_type, &ScaleTempStep, 1, critical); // read scaling frequency
  return new Woodcock(ScaleTempStep);
}

double Woodcock::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::list) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Woodcock::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::vector) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Woodcock::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::set) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double Woodcock::doScaleAtoms(unsigned int step,double ActualTemp,ForwardIterator begin,ForwardIterator end){
  double ekin=0;
  if ((ScaleTempStep > 0) && ((step-1) % ScaleTempStep == 0)) {
    if (fabs(ActualTemp) > MYEPSILON) {
      double ScaleTempFactor = sqrt(getContainer().TargetTemp/ActualTemp);
      LOG(2,  "Applying Woodcock thermostat...");
      double ekin;
      for (ForwardIterator iter = begin; iter!=end;++iter){
        Vector U = (*iter)->getAtomicVelocityAtStep(step);
        if ((*iter)->getFixedIon() == 0){ // even FixedIon moves, only not by other's forces
          U *= ScaleTempFactor;
          ekin += 0.5*(*iter)->getType()->getMass() * U.NormSquared();
        }
        (*iter)->setAtomicVelocityAtStep(step, U);
      }
    }
  }
  return ekin;
}

std::string Woodcock::name(){
  return ThermostatTraits<Woodcock>::name;
}

std::string Woodcock::writeParams(){
  std::stringstream sstr;
  sstr << ScaleTempStep;
  return sstr.str();
}
