/*
 * FragmentationLongRangeResults.hpp
 *
 *  Created on: Aug 31, 2012
 *      Author: heber
 */

#ifndef FRAGMENTATIONLONGRANGERESULTS_HPP_
#define FRAGMENTATIONLONGRANGERESULTS_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <vector>

#ifdef HAVE_JOBMARKET
#include "JobMarket/types.hpp"
#else
typedef size_t JobId_t;
#endif

#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Summation/IndexSet.hpp"
#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/SubsetMap.hpp"

#include "Fragmentation/Summation/Containers/MPQCData.hpp"
#include "Fragmentation/Summation/Containers/MPQCDataMap.hpp"
#include "Fragmentation/Summation/Containers/VMGData.hpp"
#include "Fragmentation/Summation/Containers/VMGDataMap.hpp"
#include "Fragmentation/Summation/SetValues/SamplingGrid.hpp"
#include "Fragmentation/Summation/SetValues/IndexedVectors.hpp"

/** FragmentationLongRangeResults contains the summed up results per level resulting
 * from the fragmentation of the molecular system and clever combination of
 * fragmentary energies, forces, timings, and so on.
 *
 * This structure is mostly a storage wherein the summed up results are
 * contained for subsequent pretty printing and so on.
 *
 */
struct FragmentationLongRangeResults
{
  /** Constructor for class FragmentationLongRangeResults, based on KeySets.
   *
   * @param fragmentData results from short-range fragment calculations
   * @param longrangeData results from long-range fragment calculations
   * @param _KeySet KeySets of all (non-hydrogen) atoms
   * @param _ForceKeySet KeySets of all atoms except those added by saturation
   */
  FragmentationLongRangeResults(
      const std::map<JobId_t,MPQCData> &fragmentData,
      std::map<JobId_t,VMGData> &longrangeData,
      const KeySetsContainer& _KeySet,
      const KeySetsContainer& _ForceKeySet);

  /** Performs the summation and fills all result vectors.
   *
   * @param fragmentData results from short-range fragment calculations
   * @param longrangeData results from long-range fragment calculations
   * @param fullsolutionData long-range solution
   * @param full_sample sampled density
   * @param _zero_globalgrid zero instance to use in summation, i.e. to
   *        enforce coarser grid on the global level for both potential
   *        grids
   * @param _implicit_charges_indices indices of implicit charges that do not
   *        belong to other fragments (we obtain forces for those from the
   *        long-range calculation but we lack the indices)
   */
  void operator()(
      const std::map<JobId_t,MPQCData> &fragmentData,
      std::map<JobId_t,VMGData> &longrangeData,
      const std::vector<VMGData> &fullsolutionData,
      const std::vector<SamplingGrid> &full_sample,
      const SamplingGrid &_zero_globalgrid,
      const IndexedVectors::indices_t &_implicit_charges_indices = IndexedVectors::indices_t());

  size_t getMaxLevel() const {
    return MaxLevel;
  }

  IndexSetContainer::Container_t getContainer() const {
    return container->getContainer();
  }

  const KeySetsContainer& getKeySet() const {
    return KeySet;
  }

  const KeySetsContainer& getForceKeySet() const {
    return ForceKeySet;
  }

  const bool hasLongRangeForces() const
  { return hasForces; }

private:
  void initLookups(
      const std::map<JobId_t,MPQCData> &fragmentData,
      std::map<JobId_t,VMGData> &longrangeData
      );

private:
  std::map< JobId_t, size_t > MPQCMatrixNrLookup;
  std::map< JobId_t, size_t > VMGMatrixNrLookup;
  KeySetsContainer KeySet;
  KeySetsContainer ForceKeySet;
  IndexSetContainer::ptr container;
  SubsetMap::ptr subsetmap;
  //!> indicates whether the longrange results contain VMG forces data or not
  bool hasForces;

public:
  //!> results per level of summed up sampled grid charge
  std::vector<MPQCDataGridMap_t> Result_Grid_fused;
  //!> results per level of summed up long range energies
  std::vector<VMGDataMap_t> Result_LongRange_fused;
  //!> results per level of summed up long range forces
  std::vector<VMGDataForceMap_t> Result_ForceLongRange_fused;
  //!> results per level of summed up long range potential grids
  std::vector<VMGDataGridMap_t> Result_GridLongRange_fused;
  //!> results per level of summed up long range true energy
  std::vector<VMGDataLongRangeMap_t> Result_LongRangeIntegrated_fused;
  //!> results per level of summed up long range true forces
  std::vector<VMGDataLongRangeForceMap_t> Result_ForcesLongRangeIntegrated_fused;

  //!> results per IndexSet of summed up sampled grid charge
  std::map<IndexSet::ptr, std::pair<MPQCDataGridMap_t,MPQCDataGridMap_t> > Result_perIndexSet_Grid;
  //!> results per IndexSet of summed up long range energies
  std::map<IndexSet::ptr, std::pair<VMGDataMap_t, VMGDataMap_t> > Result_perIndexSet_LongRange;
  //!> results per IndexSet of summed up long range forces
  std::map<IndexSet::ptr, std::pair<VMGDataForceMap_t, VMGDataForceMap_t> > Result_perIndexSet_LongRange_Force;
  //!> results per IndexSet of summed up long range potential grids
  std::map<IndexSet::ptr, std::pair<VMGDataGridMap_t, VMGDataGridMap_t> > Result_perIndexSet_LongRange_Grid;
  // we don't need the map pendant for Result_LongRangeIntegrated_fused, as this
  // quantity makes sense only level-wise

private:
  //!> maximum level of summation
  size_t MaxLevel;
};


#endif /* FRAGMENTATIONLONGRANGERESULTS_HPP_ */
