/*
 * MapOfActions.cpp
 *
 *  Created on: 10.05.2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

using namespace std;

#include "Actions/MapOfActions.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Helpers/Assert.hpp"
#include "Patterns/Singleton_impl.hpp"

#include <boost/lexical_cast.hpp>
#include <boost/optional.hpp>
#include <boost/program_options.hpp>

#include <iostream>

#include "atom.hpp"
#include "Box.hpp"
#include "CommandLineParser.hpp"
#include "element.hpp"
#include "log.hpp"
#include "Matrix.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "vector.hpp"
#include "verbose.hpp"

#include "Actions/ActionRegistry.hpp"
#include "Actions/AnalysisAction/MolecularVolumeAction.hpp"
#include "Actions/AnalysisAction/PairCorrelationAction.hpp"
#include "Actions/AnalysisAction/PointCorrelationAction.hpp"
#include "Actions/AnalysisAction/PrincipalAxisSystemAction.hpp"
#include "Actions/AnalysisAction/SurfaceCorrelationAction.hpp"
#include "Actions/AtomAction/AddAction.hpp"
#include "Actions/AtomAction/ChangeElementAction.hpp"
#include "Actions/AtomAction/RemoveAction.hpp"
#include "Actions/CmdAction/BondLengthTableAction.hpp"
#include "Actions/CmdAction/ElementDbAction.hpp"
#include "Actions/CmdAction/FastParsingAction.hpp"
#include "Actions/CmdAction/HelpAction.hpp"
#include "Actions/CmdAction/VerboseAction.hpp"
#include "Actions/CmdAction/VersionAction.hpp"
#include "Actions/FragmentationAction/DepthFirstSearchAction.hpp"
#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"
#include "Actions/FragmentationAction/FragmentationAction.hpp"
#include "Actions/MoleculeAction/BondFileAction.hpp"
#include "Actions/MoleculeAction/ChangeNameAction.hpp"
#include "Actions/MoleculeAction/FillWithMoleculeAction.hpp"
#include "Actions/MoleculeAction/LinearInterpolationofTrajectoriesAction.hpp"
#include "Actions/MoleculeAction/RotateToPrincipalAxisSystemAction.hpp"
#include "Actions/MoleculeAction/SaveAdjacencyAction.hpp"
#include "Actions/MoleculeAction/SaveBondsAction.hpp"
#include "Actions/MoleculeAction/SaveTemperatureAction.hpp"
#include "Actions/MoleculeAction/SuspendInWaterAction.hpp"
#include "Actions/MoleculeAction/TranslateAction.hpp"
#include "Actions/MoleculeAction/VerletIntegrationAction.hpp"
#include "Actions/ParserAction/LoadXyzAction.hpp"
#include "Actions/ParserAction/SaveXyzAction.hpp"
#include "Actions/SelectionAction/AllAtomsAction.hpp"
#include "Actions/SelectionAction/AllMoleculesAction.hpp"
#include "Actions/SelectionAction/AtomByIdAction.hpp"
#include "Actions/SelectionAction/MoleculeByIdAction.hpp"
#include "Actions/SelectionAction/NotAllAtomsAction.hpp"
#include "Actions/SelectionAction/NotAllMoleculesAction.hpp"
#include "Actions/SelectionAction/NotAtomByIdAction.hpp"
#include "Actions/SelectionAction/NotMoleculeByIdAction.hpp"
#include "Actions/TesselationAction/ConvexEnvelopeAction.hpp"
#include "Actions/TesselationAction/NonConvexEnvelopeAction.hpp"
#include "Actions/WorldAction/AddEmptyBoundaryAction.hpp"
#include "Actions/WorldAction/BoundInBoxAction.hpp"
#include "Actions/WorldAction/CenterInBoxAction.hpp"
#include "Actions/WorldAction/CenterOnEdgeAction.hpp"
#include "Actions/WorldAction/ChangeBoxAction.hpp"
#include "Actions/WorldAction/InputAction.hpp"
#include "Actions/WorldAction/OutputAction.hpp"
#include "Actions/WorldAction/RemoveSphereOfAtomsAction.hpp"
#include "Actions/WorldAction/RepeatBoxAction.hpp"
#include "Actions/WorldAction/ScaleBoxAction.hpp"
#include "Actions/WorldAction/SetDefaultNameAction.hpp"
#include "Actions/WorldAction/SetGaussianBasisAction.hpp"
#include "Actions/WorldAction/SetOutputFormatsAction.hpp"
#include "Actions/Values.hpp"

void validate(boost::any& v, const std::vector<std::string>& values, VectorValue *, int)
{
  VectorValue VV;
  std::vector<std::string> components;

  // split comma-separated values
  if (values.size() != 1) {
    cerr <<  "Not one vector but " << components.size() << " given " << endl;
    throw boost::program_options::validation_error("Unequal to one vector given");
  }
  std::string argument(values.at(0));
  std::string::iterator Aiter = argument.begin();
  std::string::iterator Biter = argument.begin();
  for (; Aiter != argument.end(); ++Aiter) {
    if (*Aiter == ',') {
      components.push_back(string(Biter,Aiter));
      do {
        Aiter++;
      } while (*Aiter == ' ' || *Aiter == '\t');
      Biter = Aiter;
    }
  }
  components.push_back(string(Biter,argument.end()));

  if (components.size() != 3) {
    cerr <<  "Specified vector does not have three components but " << components.size() << endl;
    throw boost::program_options::validation_error("Specified vector does not have three components");
  }
  VV.x = boost::lexical_cast<double>(components.at(0));
  VV.y = boost::lexical_cast<double>(components.at(1));
  VV.z = boost::lexical_cast<double>(components.at(2));
  v = boost::any(VectorValue(VV));
}

void validate(boost::any& v, const std::vector<std::string>& values, BoxValue *, int)
{
  BoxValue BV;
  std::vector<std::string> components;

  // split comma-separated values
  if (values.size() != 1) {
    cerr <<  "Not one vector but " << components.size() << " given " << endl;
    throw boost::program_options::validation_error("Unequal to one vector given");
  }
  std::string argument(values.at(0));
  std::string::iterator Aiter = argument.begin();
  std::string::iterator Biter = argument.begin();
  for (; Aiter != argument.end(); ++Aiter) {
    if (*Aiter == ',') {
      components.push_back(string(Biter,Aiter));
      do {
        Aiter++;
      } while (*Aiter == ' ' || *Aiter == '\t');
      Biter = Aiter;
    }
  }
  components.push_back(string(Biter,argument.end()));

  if (components.size() != 6) {
    cerr <<  "Specified vector does not have three components but " << components.size() << endl;
    throw boost::program_options::validation_error("Specified symmetric box matrix does not have six components");
  }
  BV.xx = boost::lexical_cast<double>(components.at(0));
  BV.yx = boost::lexical_cast<double>(components.at(1));
  BV.yy = boost::lexical_cast<double>(components.at(2));
  BV.zx = boost::lexical_cast<double>(components.at(3));
  BV.zy = boost::lexical_cast<double>(components.at(4));
  BV.zz = boost::lexical_cast<double>(components.at(5));
  v = boost::any(BoxValue(BV));
}

/** Constructor of class MapOfActions.
 *
 */
MapOfActions::MapOfActions()
{
  // initialise lookup map
  CmdParserLookup[&generic] = &(CommandLineParser::getInstance().generic);
  CmdParserLookup[&config] = &(CommandLineParser::getInstance().config);
  CmdParserLookup[&hidden] = &(CommandLineParser::getInstance().hidden);
  CmdParserLookup[&visible] = &(CommandLineParser::getInstance().visible);

  // keys for actions
  DescriptionMap["add-atom"] = "add atom of specified element";
  DescriptionMap["bond-table"] = "setting name of the bond length table file";
  DescriptionMap["bond-file"] = "name of the bond file";
  DescriptionMap["boundary"] = "change box to add an empty boundary around all atoms";
  DescriptionMap["bound-in-box"] = "bound all atoms in the domain";
  DescriptionMap["center-edge"] = "center edge of all atoms on (0,0,0)";
  DescriptionMap["center-in-box"] = "center all atoms in the domain";
  DescriptionMap["change-box"] = "change the symmetrc matrix of the simulation domain";
  DescriptionMap["change-element"] = "change the element of an atom";
  DescriptionMap["change-molname"] = "change the name of a molecule";
  DescriptionMap["convex-envelope"] = "create the convex envelope for a molecule";
  DescriptionMap["default-molname"] = "set the default name of new molecules";
  DescriptionMap["depth-first-search"] = "Depth-First Search analysis of the molecular system";
  DescriptionMap["element-db"] = "setting the path where the element databases can be found";
  DescriptionMap["fastparsing"] = "setting whether trajectories shall be parsed completely (n) or just first step (y)";
  DescriptionMap["fill-molecule"] = "fill empty space of box with a filler molecule";
  DescriptionMap["fragment-mol"] = "create for a given molecule into fragments up to given order";
  DescriptionMap["help"] = "Give this help screen";
  DescriptionMap["input"] = "specify input files";
  DescriptionMap["linear-interpolate"] = "linear interpolation in discrete steps between start and end position of a molecule";
  DescriptionMap["molecular-volume"] = "calculate the volume of a given molecule";
  DescriptionMap["nonconvex-envelope"] = "create the non-convex envelope for a molecule";
  DescriptionMap["output"] = "write output files";
  DescriptionMap["set-output"] = "specify output formats";
  DescriptionMap["pair-correlation"] = "pair correlation analysis between two elements";
  DescriptionMap["parse-xyz"] = "parse xyz file into World";
  DescriptionMap["point-correlation"] = "pair correlation analysis between element and point";
  DescriptionMap["principal-axis-system"] = "calculate the principal axis system of the specified molecule";
  DescriptionMap["remove-atom"] = "remove a specified atom";
  DescriptionMap["remove-sphere"] = "remove sphere of atoms of around a specified atom";
  DescriptionMap["repeat-box"] = "create periodic copies of the simulation box per axis";
  DescriptionMap["rotate-to-pas"] = "calculate the principal axis system of the specified molecule and rotate specified axis to align with main axis";
  DescriptionMap["save-adjacency"] = "name of the adjacency file to write to";
  DescriptionMap["save-bonds"] = "name of the bonds file to write to";
  DescriptionMap["save-temperature"] = "name of the temperature file to write to";
  DescriptionMap["SaveXyz"] = "save world as xyz file";
  DescriptionMap["scale-box"] = "scale box and atomic positions inside";
  DescriptionMap["set-basis"] = "set the name of the gaussian basis set for MPQC";
  DescriptionMap["set-output"] = "specify output formats";
  DescriptionMap["subgraph-dissect"] = "dissect the molecular system into molecules representing disconnected subgraphs";
  DescriptionMap["surface-correlation"] = "pair correlation analysis between element and surface";
  DescriptionMap["suspend-in-water"] = "suspend the given molecule in water such that in the domain the mean density is as specified";
  DescriptionMap["translate-mol"] = "translate molecule by given vector";
  DescriptionMap["verbose"] = "set verbosity level";
  DescriptionMap["verlet-integrate"] = "perform verlet integration of a given force file";
  DescriptionMap["version"] = "show version";
  // keys for values
  DescriptionMap["bin-output-file"] = "name of the bin output file";
  DescriptionMap["bin-end"] = "start of the last bin";
  DescriptionMap["bin-start"] = "start of the first bin";
  DescriptionMap["bin-width"] = "width of the bins";
  DescriptionMap["convex-file"] = "filename of the non-convex envelope";
  DescriptionMap["distance"] = "distance in space";
  DescriptionMap["distances"] = "list of three of distances in space, one for each axis direction";
  DescriptionMap["DoRotate"] = "whether to rotate or just report angles";
  DescriptionMap["element"] = "single element";
  DescriptionMap["elements"] = "set of elements";
  DescriptionMap["end-step"] = "last or end step";
  DescriptionMap["id-mapping"] = "whether the identity shall be used in mapping atoms onto atoms or some closest distance measure shall be used";
  DescriptionMap["input"] = "name of input file";
  DescriptionMap["length"] = "length in space";
  DescriptionMap["lengths"] = "list of three of lengths in space, one for each axis direction";
  DescriptionMap["MaxDistance"] = "maximum distance in space";
  DescriptionMap["molecule-by-id"] = "index of a molecule";
  DescriptionMap["nonconvex-file"] = "filename of the non-convex envelope";
  DescriptionMap["order"] = "order of a discretization, dissection, ...";
  DescriptionMap["output-file"] = "name of the output file";
  DescriptionMap["periodic"] = "system is constraint to periodic boundary conditions (y/n)";
  DescriptionMap["position"] = "position in R^3 space";
  DescriptionMap["select-all-atoms"] = "select all atoms";
  DescriptionMap["select-all-molecules"] = "select all molecules";
  DescriptionMap["select-atom-by-id"] = "select an atom by index";
  DescriptionMap["select-molecule-by-id"] = "select a molecule by index";
  DescriptionMap["start-step"] = "first or start step";
  DescriptionMap["unselect-all-atoms"] = "unselect all atoms";
  DescriptionMap["unselect-all-molecules"] = "unselect all molecules";
  DescriptionMap["unselect-atom-by-id"] = "unselect an atom by index";
  DescriptionMap["unselect-molecule-by-id"] = "unselect a molecule by index";

  // short forms for the actions
  ShortFormMap["add-atom"] = "a";
  ShortFormMap["bond-table"] = "g";
  ShortFormMap["bond-file"] = "A";
  ShortFormMap["boundary"] = "c";
  ShortFormMap["change-box"] = "B";
  ShortFormMap["center-edge"] = "O";
  ShortFormMap["center-in-box"] = "b";
  ShortFormMap["change-element"] = "E";
//  ShortFormMap["convex-envelope"] = "x";
  ShortFormMap["default-molname"] = "X";
  ShortFormMap["depth-first-search"] = "D";
  ShortFormMap["element-db"] = "e";
  ShortFormMap["fastparsing"] = "n";
  ShortFormMap["fill-molecule"] = "F";
  ShortFormMap["fragment-mol"] = "f";
  ShortFormMap["help"] = "h";
  ShortFormMap["input"] = "i";
  ShortFormMap["linear-interpolate"] = "L";
  ShortFormMap["nonconvex-envelope"] = "N";
//  ShortFormMap["output"] = "o";
//  ShortFormMap["pair-correlation"] = "C";
  ShortFormMap["parse-xyz"] = "p";
  ShortFormMap["remove-atom"] = "r";
  ShortFormMap["remove-sphere"] = "R";
  ShortFormMap["repeat-box"] = "d";
  ShortFormMap["rotate-to-pas"] = "m";
  ShortFormMap["save-adjacency"] = "J";
  ShortFormMap["save-bonds"] = "j";
  ShortFormMap["save-temperature"] = "S";
  ShortFormMap["scale-box"] = "s";
  ShortFormMap["set-basis"] = "M";
  ShortFormMap["set-output"] = "o";
  ShortFormMap["subgraph-dissect"] = "I";
  ShortFormMap["suspend-in-water"] = "u";
  ShortFormMap["translate-mol"] = "t";
  ShortFormMap["verbose"] = "v";
  ShortFormMap["verlet-integrate"] = "P";
  ShortFormMap["version"] = "V";

  // value types for the actions
  TypeMap["add-atom"] = &typeid(element);
  TypeMap["bond-file"] = &typeid(std::string);
  TypeMap["bond-table"] = &typeid(std::string);
  TypeMap["boundary"] = &typeid(VectorValue);
  TypeMap["center-in-box"] = &typeid(BoxValue);
  TypeMap["change-box"] = &typeid(BoxValue);
  TypeMap["change-element"] = &typeid(element);
  TypeMap["change-molname"] = &typeid(std::string);
  TypeMap["convex-envelope"] = &typeid(void);
  TypeMap["default-molname"] = &typeid(std::string);
  TypeMap["depth-first-search"] = &typeid(double);
  TypeMap["element-db"] = &typeid(std::string);
  TypeMap["fastparsing"] = &typeid(bool);
  TypeMap["fill-molecule"] = &typeid(std::string);
  TypeMap["fragment-mol"] = &typeid(std::string);
  TypeMap["input"] = &typeid(std::string);
  TypeMap["linear-interpolate"] = &typeid(std::string);
  TypeMap["molecular-volume"] = &typeid(molecule);
  TypeMap["nonconvex-envelope"] = &typeid(double);
  TypeMap["output"] = &typeid(void);
  TypeMap["parse-xyz"] = &typeid(std::string);
  TypeMap["pair-correlation"] = &typeid(void);
  TypeMap["point-correlation"] = &typeid(void);
  TypeMap["principal-axis-system"] = &typeid(molecule);
  TypeMap["remove-atom"] = &typeid(void);
  TypeMap["remove-sphere"] = &typeid(double);
  TypeMap["repeat-box"] = &typeid(VectorValue);
  TypeMap["rotate-to-pas"] = &typeid(molecule);
  TypeMap["save-adjacency"] = &typeid(std::string);
  TypeMap["save-bonds"] = &typeid(std::string);
  TypeMap["save-temperature"] = &typeid(std::string);
  TypeMap["scale-box"] = &typeid(VectorValue);
  TypeMap["set-basis"] = &typeid(std::string);
  TypeMap["set-output"] = &typeid(std::vector<std::string>);
  TypeMap["subgraph-dissect"] = &typeid(void);
  TypeMap["surface-correlation"] = &typeid(void);
  TypeMap["suspend-in-water"] = &typeid(double);
  TypeMap["translate-mol"] = &typeid(VectorValue);
  TypeMap["verlet-integrate"] = &typeid(std::string);
  TypeMap["verbose"] = &typeid(int);

  // value types for the values
  TypeMap["bin-output-file"] = &typeid(std::string);
  TypeMap["bin-end"] = &typeid(double);
  TypeMap["bin-start"] = &typeid(double);
  TypeMap["bin-width"] = &typeid(double);
  TypeMap["convex-file"] = &typeid(std::string);
  TypeMap["distance"] = &typeid(double);
  TypeMap["distances"] = &typeid(VectorValue);
  TypeMap["DoRotate"] = &typeid(bool);
  TypeMap["element"] = &typeid(element);
  TypeMap["elements"] = &typeid(std::vector<element *>);
  TypeMap["end-step"] = &typeid(int);
  TypeMap["id-mapping"] = &typeid(bool);
  TypeMap["length"] = &typeid(double);
  TypeMap["lengths"] = &typeid(VectorValue);
  TypeMap["MaxDistance"] = &typeid(double);
  TypeMap["molecule-by-id"] = &typeid(molecule);
  TypeMap["nonconvex-file"] = &typeid(std::string);
  TypeMap["order"] = &typeid(int);
  TypeMap["output-file"] = &typeid(std::string);
  TypeMap["periodic"] = &typeid(bool);
  TypeMap["position"] = &typeid(VectorValue);
  TypeMap["select-all-atoms"] = &typeid(void);
  TypeMap["select-all-molecules"] = &typeid(void);
  TypeMap["select-atom-by-id"] = &typeid(atom);
  TypeMap["select-molecule-by-id"] = &typeid(molecule);
  TypeMap["start-step"] = &typeid(int);
  TypeMap["unselect-all-atoms"] = &typeid(void);
  TypeMap["unselect-all-molecules"] = &typeid(void);
  TypeMap["unselect-atom-by-id"] = &typeid(atom);
  TypeMap["unselect-molecule-by-id"] = &typeid(molecule);

  TypeEnumMap[&typeid(void)] = None;
  TypeEnumMap[&typeid(bool)] = Boolean;
  TypeEnumMap[&typeid(int)] = Integer;
  TypeEnumMap[&typeid(std::vector<int>)] = ListOfIntegers;
  TypeEnumMap[&typeid(double)] = Double;
  TypeEnumMap[&typeid(std::vector<double>)] = ListOfDoubles;
  TypeEnumMap[&typeid(std::string)] = String;
  TypeEnumMap[&typeid(std::vector<std::string>)] = ListOfStrings;
  TypeEnumMap[&typeid(VectorValue)] = Vector;
  TypeEnumMap[&typeid(std::vector<VectorValue>)] = ListOfVectors;
  TypeEnumMap[&typeid(BoxValue)] = Box;
  TypeEnumMap[&typeid(molecule)] = Molecule;
  TypeEnumMap[&typeid(std::vector<molecule *>)] = ListOfMolecules;
  TypeEnumMap[&typeid(atom)] = Atom;
  TypeEnumMap[&typeid(std::vector<atom *>)] = ListOfAtoms;
  TypeEnumMap[&typeid(element)] = Element;
  TypeEnumMap[&typeid(std::vector<element *>)] = ListOfElements;

  // default values for any action that needs one (always string!)
  CurrentValue["bin-width"] = "0.5";
  CurrentValue["fastparsing"] = "0";
  CurrentValue["periodic"] = "0";

  // put action into each menu category
  MenuDescription["analysis"] = pair<std::string,std::string>("Analysis (pair correlation, volume)", "Analysis");
  MenuDescription["atom"] = pair<std::string,std::string>("Edit atoms", "Atoms");
  MenuDescription["command"] = pair<std::string,std::string>("Configuration", "configuration options");
  MenuDescription["fragmentation"] = pair<std::string,std::string>("Fragmentation", "Fragmentation");
  MenuDescription["molecule"] = pair<std::string,std::string>("Parse files into system", "Molecules");
  MenuDescription["parser"] = pair<std::string,std::string>("Edit molecules (load, parse, save)", "Input/Output");
  MenuDescription["selection"] = pair<std::string,std::string>("Select atoms/molecules", "Selection");
  MenuDescription["tesselation"] = pair<std::string,std::string>("Tesselate molecules", "Tesselation");
  MenuDescription["world"] = pair<std::string,std::string>("Edit world", "Globals");

  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "molecular-volume") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "pair-correlation") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "point-correlation") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "surface-correlation") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("analysis", "principal-axis-system") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("atom", "add-atom") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("atom", "change-element") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("atom", "remove-atom") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "bond-table") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "element-db") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "fastparsing") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "verbose") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("command", "version") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("fragmentation", "depth-first-search") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("fragmentation", "fragment-mol") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("fragmentation", "subgraph-dissect") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "bond-file") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "change-molname") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "fill-molecule") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "linear-interpolate") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "rotate-to-pas") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "save-adjacency") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "save-bonds") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "save-temperature") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "suspend-in-water") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "translate-mol") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("molecule", "verlet-integrate") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("parser", "parse-xyz") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("parser", "SaveXyz") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("selection", "select-atom-by-id") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("selection", "select-molecule-by-id") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("selection", "unselect-atom-by-id") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("selection", "unselect-molecule-by-id") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("tesselation", "convex-envelope") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("tesselation", "nonconvex-envelope") );

  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "boundary") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "bound-in-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "center-in-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "center-edge") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "change-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "input") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "output") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "remove-sphere") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "repeat-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "scale-box") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "default-molname") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "set-basis") );
  MenuContainsActionMap.insert( pair<std::string, std::string> ("world", "set-output") );

  // put actions into command line category
	generic.insert("add-atom");
  generic.insert("bond-file");
	generic.insert("bond-table");
  generic.insert("boundary");
//  generic.insert("bound-in-box");
  generic.insert("center-edge");
  generic.insert("center-in-box");
	generic.insert("change-box");
//  generic.insert("change-molname");
	generic.insert("change-element");
  generic.insert("convex-envelope");
	generic.insert("default-molname");
	generic.insert("depth-first-search");
	generic.insert("element-db");
	generic.insert("fastparsing");
  generic.insert("fill-molecule");
  generic.insert("fragment-mol");
  generic.insert("help");
  generic.insert("input");
  generic.insert("linear-interpolate");
//  generic.insert("molecular-volume");
  generic.insert("nonconvex-envelope");
  generic.insert("output");
	generic.insert("pair-correlation");
	generic.insert("parse-xyz");
  generic.insert("point-correlation");
//  generic.insert("principal-axis-system");
  generic.insert("remove-atom");
  generic.insert("remove-sphere");
  generic.insert("repeat-box");
  generic.insert("rotate-to-pas");
	generic.insert("save-adjacency");
  generic.insert("save-bonds");
  generic.insert("save-temperature");
  generic.insert("scale-box");
  generic.insert("select-all-atoms");
  generic.insert("select-all-molecules");
  generic.insert("select-atom-by-id");
  generic.insert("select-molecule-by-id");
  generic.insert("set-basis");
  generic.insert("set-output");
	generic.insert("subgraph-dissect");
  generic.insert("surface-correlation");
  generic.insert("suspend-in-water");
  generic.insert("translate-mol");
  generic.insert("unselect-all-atoms");
  generic.insert("unselect-all-molecules");
  generic.insert("unselect-atom-by-id");
  generic.insert("unselect-molecule-by-id");
	generic.insert("verbose");
  generic.insert("verlet-integrate");
	generic.insert("version");

    // positional arguments
  generic.insert("input");

    // hidden arguments
  hidden.insert("bin-end");
  hidden.insert("bin-output-file");
  hidden.insert("bin-start");
  hidden.insert("bin-width");
  hidden.insert("convex-file");
  hidden.insert("distance");
  hidden.insert("DoRotate");
  hidden.insert("distances");
  hidden.insert("element");
  hidden.insert("elements");
  hidden.insert("end-step");
  hidden.insert("id-mapping");
  hidden.insert("lengths");
  hidden.insert("MaxDistance");
  hidden.insert("molecule-by-id");
  hidden.insert("nonconvex-file");
  hidden.insert("order");
  hidden.insert("output-file");
  hidden.insert("periodic");
  hidden.insert("position");
  hidden.insert("start-step");
}

/** Destructor of class MapOfActions.
 *
 */
MapOfActions::~MapOfActions()
{
  DescriptionMap.clear();
}

void MapOfActions::queryCurrentValue(const char * name, class atom * &_T)
{
  int atomID = -1;
  if (typeid( atom ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    atomID = lexical_cast<int>(CurrentValue[name].c_str());
    CurrentValue.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
  _T = World::getInstance().getAtom(AtomById(atomID));
}

void MapOfActions::queryCurrentValue(const char * name, class element * &_T)  {
  int Z = -1;
  if (typeid( element ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    Z = lexical_cast<int>(CurrentValue[name].c_str());
    CurrentValue.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
  _T = World::getInstance().getPeriode()->FindElement(Z);
}

void MapOfActions::queryCurrentValue(const char * name, class molecule * &_T) {
  int molID = -1;
  if (typeid( molecule ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    molID = lexical_cast<int>(CurrentValue[name].c_str());
    CurrentValue.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
  _T = World::getInstance().getMolecule(MoleculeById(molID));
}

void MapOfActions::queryCurrentValue(const char * name, class Box &_T) {
  Matrix M;
  double tmp;
  if (typeid( BoxValue ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValue[name]);
    stream >> tmp;
    M.set(0,0,tmp);
    stream >> tmp;
    M.set(0,1,tmp);
    M.set(1,0,tmp);
    stream >> tmp;
    M.set(0,2,tmp);
    M.set(2,0,tmp);
    stream >> tmp;
    M.set(1,1,tmp);
    stream >> tmp;
    M.set(1,2,tmp);
    M.set(2,1,tmp);
    stream >> tmp;
    M.set(2,2,tmp);
    _T = M;
    CurrentValue.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::queryCurrentValue(const char * name, class Vector &_T) {
  if (typeid( VectorValue ) == *TypeMap[name]) {
    std::istringstream stream(CurrentValue[name]);
    CurrentValue.erase(name);
    stream >> _T[0];
    stream >> _T[1];
    stream >> _T[2];
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::queryCurrentValue(const char * name, std::vector<atom *>&_T)
{
  int atomID = -1;
  atom *Walker = NULL;
  if (typeid( std::vector<atom *> ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValue[name]);
    CurrentValue.erase(name);
    while (!stream.fail()) {
      stream >> atomID >> ws;
      Walker = World::getInstance().getAtom(AtomById(atomID));
      if (Walker != NULL)
        _T.push_back(Walker);
      atomID = -1;
      Walker = NULL;
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::queryCurrentValue(const char * name, std::vector<element *>&_T)
{
  int Z = -1;
  element *elemental = NULL;
  if (typeid( std::vector<element *> ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValue[name]);
    CurrentValue.erase(name);
    while (!stream.fail()) {
      stream >> Z >> ws;
      elemental = World::getInstance().getPeriode()->FindElement(Z);
      if (elemental != NULL)
        _T.push_back(elemental);
      Z = -1;
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::queryCurrentValue(const char * name, std::vector<molecule *>&_T)
{
  int molID = -1;
  molecule *mol = NULL;
  if (typeid( std::vector<molecule *> ) == *TypeMap[name]) {
    if (CurrentValue.find(name) == CurrentValue.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValue[name]);
    CurrentValue.erase(name);
    while (!stream.fail()) {
      stream >> molID >> ws;
      mol = World::getInstance().getMolecule(MoleculeById(molID));
      if (mol != NULL)
        _T.push_back(mol);
      molID = -1;
      mol = NULL;
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}


void MapOfActions::setCurrentValue(const char * name, class atom * &_T)
{
  if (typeid( atom ) == *TypeMap[name]) {
    std::ostringstream stream;
    stream << _T->getId();
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, class element * &_T)
{
  if (typeid( element ) == *TypeMap[name]) {
    std::ostringstream stream;
    stream << _T->Z;
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, class molecule * &_T)
{
  if (typeid( molecule ) == *TypeMap[name]) {
    std::ostringstream stream;
    stream << _T->getId();
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, class Box &_T)
{
  const Matrix &M = _T.getM();
  if (typeid( BoxValue ) == *TypeMap[name]) {
    std::ostringstream stream;
    stream << M.at(0,0) << " ";
    stream << M.at(0,1) << " ";
    stream << M.at(0,2) << " ";
    stream << M.at(1,1) << " ";
    stream << M.at(1,2) << " ";
    stream << M.at(2,2) << " ";
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, class Vector &_T)
{
  if (typeid( VectorValue ) == *TypeMap[name]){
    std::ostringstream stream;
    stream << _T[0] << " ";
    stream << _T[1] << " ";
    stream << _T[2] << " ";
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, std::vector<atom *>&_T)
{
  if (typeid( std::vector<atom *> ) == *TypeMap[name]) {
    std::ostringstream stream;
    for (std::vector<atom *>::iterator iter = _T.begin(); iter != _T.end(); ++iter) {
      stream << (*iter)->getId() << " ";
    }
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, std::vector<element *>&_T)
{
  if (typeid( std::vector<element *> ) == *TypeMap[name]) {
    std::ostringstream stream;
    for (std::vector<element *>::iterator iter = _T.begin(); iter != _T.end(); ++iter) {
      stream << (*iter)->Z << " ";
    }
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void MapOfActions::setCurrentValue(const char * name, std::vector<molecule *>&_T)
{
  if (typeid( std::vector<molecule *> ) == *TypeMap[name]) {
    std::ostringstream stream;
    for (std::vector<molecule *>::iterator iter = _T.begin(); iter != _T.end(); ++iter) {
      stream << (*iter)->getId() << " ";
    }
    CurrentValue[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}



void MapOfActions::populateActions()
{
  new AnalysisMolecularVolumeAction();
  new AnalysisPairCorrelationAction();
  new AnalysisPointCorrelationAction();
  new AnalysisPrincipalAxisSystemAction();
  new AnalysisSurfaceCorrelationAction();

  new AtomAddAction();
  new AtomChangeElementAction();
  new AtomRemoveAction();

  new CommandLineBondLengthTableAction();
  new CommandLineElementDbAction();
  new CommandLineFastParsingAction();
  new CommandLineHelpAction();
  new CommandLineVerboseAction();
  new CommandLineVersionAction();

  new FragmentationDepthFirstSearchAction();
  new FragmentationFragmentationAction();
  new FragmentationSubgraphDissectionAction();

  new MoleculeBondFileAction();
  new MoleculeChangeNameAction();
  new MoleculeFillWithMoleculeAction();
  new MoleculeLinearInterpolationofTrajectoriesAction();
  new MoleculeRotateToPrincipalAxisSystemAction();
  new MoleculeSaveAdjacencyAction();
  new MoleculeSaveBondsAction();
  new MoleculeSaveTemperatureAction();
  new MoleculeSuspendInWaterAction();
  new MoleculeTranslateAction();
  new MoleculeVerletIntegrationAction();

  new ParserLoadXyzAction();
  new ParserSaveXyzAction();

  new SelectionAllAtomsAction();
  new SelectionAllMoleculesAction();
  new SelectionAtomByIdAction();
  new SelectionMoleculeByIdAction();
  new SelectionNotAllAtomsAction();
  new SelectionNotAllMoleculesAction();
  new SelectionNotAtomByIdAction();
  new SelectionNotMoleculeByIdAction();

  new TesselationConvexEnvelopeAction();
  new TesselationNonConvexEnvelopeAction();

  new WorldAddEmptyBoundaryAction();
  new WorldBoundInBoxAction();
  new WorldCenterInBoxAction();
  new WorldCenterOnEdgeAction();
  new WorldChangeBoxAction();
  new WorldInputAction();
  new WorldOutputAction();
  new WorldRemoveSphereOfAtomsAction();
  new WorldRepeatBoxAction();
  new WorldScaleBoxAction();
  new WorldSetDefaultNameAction();
  new WorldSetGaussianBasisAction();
  new WorldSetOutputFormatsAction();
}

/** Adds all options to the CommandLineParser.
 *
 */
void MapOfActions::AddOptionsToParser()
{
  // add other options
  for (map< set<string>*, po::options_description* >::iterator ListRunner = CmdParserLookup.begin(); ListRunner != CmdParserLookup.end(); ++ListRunner) {
    for (set<string>::iterator OptionRunner = ListRunner->first->begin(); OptionRunner != ListRunner->first->end(); ++OptionRunner) {
      if (hasValue(*OptionRunner)) {
        DoLog(1) && (Log() << Verbose(1) << "Adding option " << *OptionRunner << " with type " << TypeMap[*OptionRunner]->name() << " to CommandLineParser." << endl);
           switch(TypeEnumMap[TypeMap[*OptionRunner]]) {
          default:
          case None:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(), getDescription(*OptionRunner).c_str())
              ;
            break;
          case Boolean:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  CurrentValue.find(*OptionRunner) != CurrentValue.end() ?
                        po::value< bool >()->default_value(lexical_cast<int>(CurrentValue[*OptionRunner].c_str())) :
                        po::value< bool >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Box:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value<BoxValue>(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Integer:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  CurrentValue.find(*OptionRunner) != CurrentValue.end() ?
                        po::value< int >()->default_value(lexical_cast<int>(CurrentValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfIntegers:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Double:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  CurrentValue.find(*OptionRunner) != CurrentValue.end() ?
                        po::value< double >()->default_value(lexical_cast<double>(CurrentValue[*OptionRunner].c_str())) :
                        po::value< double >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfDoubles:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<double> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case String:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  CurrentValue.find(*OptionRunner) != CurrentValue.end() ?
                        po::value< std::string >()->default_value(CurrentValue[*OptionRunner]) :
                        po::value< std::string >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfStrings:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<std::string> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Vector:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value<VectorValue>(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfVectors:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<VectorValue> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Molecule:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  CurrentValue.find(*OptionRunner) != CurrentValue.end() ?
                        po::value< int >()->default_value(lexical_cast<int>(CurrentValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfMolecules:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Atom:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  CurrentValue.find(*OptionRunner) != CurrentValue.end() ?
                        po::value< int >()->default_value(lexical_cast<int>(CurrentValue[*OptionRunner].c_str())) :
                        po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfAtoms:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case Element:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< int >(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
          case ListOfElements:
            ListRunner->second->add_options()
              (getKeyAndShortForm(*OptionRunner).c_str(),
                  po::value< vector<int> >()->multitoken(),
                  getDescription(*OptionRunner).c_str())
              ;
            break;
        }
      } else {
        DoLog(3) && (Log() << Verbose(3) << "Adding option " << *OptionRunner << " to CommandLineParser." << endl);
        ListRunner->second->add_options()
          (getKeyAndShortForm(*OptionRunner).c_str(), getDescription(*OptionRunner).c_str())
          ;
      }
    }
  }
}

/** Getter for MapOfActions:DescriptionMap.
 * Note that we assert when action does not exist in CommandLineParser::DescriptionMap.
 * \param actionname name of the action to lookup
 * \return Description of the action
 */
std::string MapOfActions::getDescription(string actionname)
{
  ASSERT(DescriptionMap.find(actionname) != DescriptionMap.end(), "Unknown action name passed to MapOfActions::getDescription");
  return DescriptionMap[actionname];
}

/** Specific Getter for a MapOfActions:ShortFormMap.
 * If action has a short for, then combination is as "actionname,ShortForm" (this is
 * the desired format for boost::program_options). If no short form exists in the map,
 * just actionname will be returned
 * Note that we assert when action does not exist in CommandLineParser::DescriptionMap.
 * \param actionname name of the action to lookup
 * \return actionname,ShortForm or Description of the action
 */
std::string MapOfActions::getKeyAndShortForm(string actionname)
{
  stringstream output;
  ASSERT(DescriptionMap.find(actionname) != DescriptionMap.end(), "Unknown action name passed to MapOfActions::getDescriptionAndShortForm");
  output << actionname;
  if (ShortFormMap.find(actionname) != DescriptionMap.end())
    output << "," << ShortFormMap[actionname];
  return output.str();
}

/** Getter for MapOfActions:ShortFormMap.
 * Note that we assert when action does not exist CommandLineParser::ShortFormMap.
 * \param actionname name of the action to lookup
 * \return ShortForm of the action
 */
std::string MapOfActions::getShortForm(string actionname)
{
  ASSERT(ShortFormMap.find(actionname) != ShortFormMap.end(), "Unknown action name passed to MapOfActions::getShortForm");
  return ShortFormMap[actionname];
}

/** Returns whether the given action needs a value or not.
 * \param actionname name of the action to look up
 * \return true - value is needed, false - no value is stored in MapOfActions::TypeMap
 */
bool MapOfActions::hasValue(string actionname)
{
  return (TypeMap.find(actionname) != TypeMap.end());
}

/** Getter for MapOfActions::TypeMap.
 * \param actionname name of the action to look up
 * \return type of the action
 */
std::string MapOfActions::getValueType(string actionname)
{
  return TypeMap[actionname]->name();
}

/** Searches whether action is registered with CommandLineParser.
 * Note that this method is only meant transitionally for ParseCommandLineOptions' removal.
 * I.e. All actions that are already handled by the new CommandLineUIFactory can be checked
 * by this function.
 * \param shortform command short form to look for
 * \return true - action has been registered, false - action has not been registered.
 */
bool MapOfActions::isShortFormPresent(string shortform)
{
  bool result = false;
  string actionname;
  for (map<std::string, std::string>::iterator ShortFormRunner = ShortFormMap.begin(); ShortFormRunner != ShortFormMap.end(); ++ShortFormRunner)
    if (ShortFormRunner->second == shortform) {
      actionname = ShortFormRunner->first;
      break;
    }
  result = result || (generic.find(actionname) != generic.end());
  result = result || (config.find(actionname) != config.end());
  result = result || (hidden.find(actionname) != hidden.end());
  result = result || (visible.find(actionname) != visible.end());
  result = result || (inputfile.find(actionname) != inputfile.end());
  return result;
}

/** Returns the inverse to MapOfActions::ShortFormMap, i.e. lookup actionname for its short form.
 * \return map from short form of action to name of action
 */
map <std::string, std::string> MapOfActions::getShortFormToActionMap()
{
  map <std::string, std::string> result;

  for (map<std::string, std::string>::iterator iter = ShortFormMap.begin(); iter != ShortFormMap.end();  ++iter)
    result[iter->second] = iter->first;

  return result;
}


CONSTRUCT_SINGLETON(MapOfActions)
