/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2017 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * RandomPerturbationAction.cpp
 *
 *  Created on: Apr 2, 2017
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Vector.hpp"

#include "Actions/UndoRedoHelpers.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomicInfo.hpp"
#include "RandomNumbers/RandomNumberGenerator.hpp"
#include "RandomNumbers/RandomNumberGeneratorFactory.hpp"
#include "World.hpp"

#include <iostream>
#include <map>
#include <string>

#include "Actions/AtomAction/RandomPerturbationAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "RandomPerturbationAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr AtomRandomPerturbationAction::performCall() {
  // check preconditions
  if (World::getInstance().countSelectedAtoms() == 0) {
    STATUS("No atoms are selected whose positions to randomly perturb.");
    return Action::failure;
  }

  // create undo state
  std::vector<AtomicInfo> Walkers;
  for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
      iter != World::getInstance().endAtomSelection();
      ++iter) {
    Walkers.push_back(AtomicInfo(*(iter->second)));
  }

  // prepare noise generator
  RandomNumberGenerator &random = RandomNumberGeneratorFactory::getInstance().makeRandomNumberGenerator();
  const double rng_min = random.min();
  const double rng_max = random.max();
  const double range = (rng_max - rng_min);
  const double offset = rng_min;

  std::vector<AtomicInfo> NewWalkers;
  for (World::AtomSelectionIterator iter = World::getInstance().beginAtomSelection(); iter != World::getInstance().endAtomSelection(); ++iter) {
    atom *first = iter->second;
    double noisecomponents[NDIM];
    for (size_t i=0;i<NDIM;++i)
      noisecomponents[i] = params.noise.get()*((random()-offset)*2./range - 1.);
    const Vector noisevector(noisecomponents);
    LOG(3, "DEBUG: Changing atom " << *first << "'s position by " << noisevector << ".");
    first->setPosition(first->getPosition() + noisevector);
    NewWalkers.push_back(AtomicInfo(*first));
  }

  AtomRandomPerturbationState *UndoState = new AtomRandomPerturbationState(Walkers, NewWalkers, params);
  return ActionState::ptr(UndoState);
}

ActionState::ptr AtomRandomPerturbationAction::performUndo(ActionState::ptr _state) {
  AtomRandomPerturbationState *state = assert_cast<AtomRandomPerturbationState*>(_state.get());

  // set stored old state
  SetAtomsFromAtomicInfo(state->Walkers);

  return ActionState::ptr(_state);
}

ActionState::ptr AtomRandomPerturbationAction::performRedo(ActionState::ptr _state){
  AtomRandomPerturbationState *state = assert_cast<AtomRandomPerturbationState*>(_state.get());

  // perturb with same noise vectors
  SetAtomsFromAtomicInfo(state->NewWalkers);

  return ActionState::ptr(_state);
}

bool AtomRandomPerturbationAction::canUndo() {
  return true;
}

bool AtomRandomPerturbationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
