/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CenterOnEdgeAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include "boost/serialization/vector.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "atom.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/MatrixContent.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <string>
#include <vector>

#include "Actions/WorldAction/CenterOnEdgeAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "CenterOnEdgeAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr WorldCenterOnEdgeAction::performCall() {
  Vector Min;
  Vector Max;

  // obtain information
  getParametersfromValueStorage();

  // create undo state
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  const RealSpaceMatrix &matrix = World::getInstance().getDomain().getM();
  oa << matrix;
  std::vector< boost::shared_ptr<Vector> > OldPositions;
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin(); AtomRunner != AllAtoms.end(); ++AtomRunner)
    OldPositions.push_back(
        boost::shared_ptr<Vector>(new Vector(
            (*AtomRunner)->getPosition()
            ))
        );

  // get maximum and minimum
  ASSERT(AllAtoms.size() > 0, "For CenteronEdge atoms must be present.");
  std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
  Min = (*AtomRunner)->getPosition();
  Max = (*AtomRunner)->getPosition();
  for (; AtomRunner != AllAtoms.end(); ++AtomRunner) {
    for (int i=0;i<NDIM;i++) {
      if ((*AtomRunner)->at(i) > Max[i])
        Max[i] = (*AtomRunner)->at(i);
      if ((*AtomRunner)->at(i) < Min[i])
        Min[i] = (*AtomRunner)->at(i);
    }
  }
  // set new box size
  RealSpaceMatrix domain;
  for (int i=0;i<NDIM;i++) {
    double tmp = Max[i]-Min[i];
    tmp = fabs(tmp)>=1. ? tmp : 1.0;
    domain.at(i,i) = tmp;
  }
  World::getInstance().setDomain(domain);
  // translate all atoms, such that Min is aty (0,0,0)
  for (std::vector<atom*>::iterator AtomRunner = AllAtoms.begin(); AtomRunner != AllAtoms.end(); ++AtomRunner)
    *(*AtomRunner) -= Min;

  // give final box size
  LOG(0, "Box domain is now " << World::getInstance().getDomain().getM());

  // create undo state
  WorldCenterOnEdgeState *UndoState =
      new WorldCenterOnEdgeState(
          undostream.str(),
          Min,
          Max,
          params
          );

  return Action::state_ptr(UndoState);
}

Action::state_ptr WorldCenterOnEdgeAction::performUndo(Action::state_ptr _state) {
  WorldCenterOnEdgeState *state = assert_cast<WorldCenterOnEdgeState*>(_state.get());

  // restore domain
  RealSpaceMatrix matrix;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> matrix;
  World::getInstance().setDomain(matrix);

  // translate all atoms back
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (vector<atom*>::iterator AtomRunner = AllAtoms.begin();
      AtomRunner != AllAtoms.end();
      ++AtomRunner)
    *(*AtomRunner) += state->Min;

  // give final box size
  LOG(0, "Box domain restored to " << World::getInstance().getDomain().getM());

  return Action::state_ptr(_state);
}

Action::state_ptr WorldCenterOnEdgeAction::performRedo(Action::state_ptr _state){
  WorldCenterOnEdgeState *state = assert_cast<WorldCenterOnEdgeState*>(_state.get());

  // set new box size
  RealSpaceMatrix rmatrix;
  for (int i=0;i<NDIM;i++) {
    double tmp = state->Max[i]-state->Min[i];
    tmp = fabs(tmp)>=1. ? tmp : 1.0;
    rmatrix.at(i,i) = tmp;
  }
  World::getInstance().setDomain(rmatrix);
  // translate all atoms, such that Min is aty (0,0,0)
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (vector<atom*>::iterator AtomRunner = AllAtoms.begin();
      AtomRunner != AllAtoms.end();
      ++AtomRunner)
    *(*AtomRunner) -= state->Min;

  // give final box size
  LOG(0, "Box domain is again " << World::getInstance().getDomain().getM());

  return Action::state_ptr(_state);
}

bool WorldCenterOnEdgeAction::canUndo() {
  return true;
}

bool WorldCenterOnEdgeAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
