#ifndef ATOMDESCRIPTOR_IMPL_HPP
#define ATOMDESCRIPTOR_IMPL_HPP

#include "Descriptors/AtomDescriptor.hpp"

/************************ Declarations of implementation Objects ************************/

/**
 * This class implements a general Base class for AtomDescriptors using the PIMPL-Idiom
 *
 * The predicate for this class is empty and should be implemented by derived classes.
 * By the predicate it is described which atoms should be picked for a given descriptor.
 */

class AtomDescriptor_impl
{
  friend class AtomDescriptor;
public:

  AtomDescriptor_impl();
  virtual ~AtomDescriptor_impl();

  /**
   * Implement this abstract Method to make a concrete AtomDescriptor pick certain Atoms
   */
  virtual bool predicate(std::pair<atomId_t,atom*>)=0;

protected:

  /**
   * This method is called when the Descriptor is used to find the first matching
   * Atom. Walks through all Atoms and stops on the first match. Can be implemented
   * when the searched Atom can be found in a more efficient way. Calculated
   * Atomdescriptors will always use this method, so no improvement there.
   */
  virtual atom* find();

  /**
   * This method is called when the Descriptor is used to find all matching Atoms.
   * Walks through all Atoms and tests the predicate on each one. A vector of all
   * matching Atoms is returned.
   */
  virtual std::vector<atom*> findAll();

  /**
   * This method is used internally to query the Set of Atoms from the world.
   * By using this method derived classes can also access the Internal World Datastructre.
   * Implemented in full in the Base Descriptor Implementation, so only this one method
   * needs to be friend with the World class.
   */
  World::AtomSet& getAtoms();
};

/************************** Universe and Emptyset *****************/

/**
 * A simple AtomDescriptor that will always match all Atoms present in the World.
 */
class AtomAllDescriptor_impl : public AtomDescriptor_impl {
public:
  AtomAllDescriptor_impl();
  virtual ~AtomAllDescriptor_impl();

  /**
   * Always returns true for any Atom
   */
  virtual bool predicate(std::pair<atomId_t,atom*>);
};


/**
 * An AtomDescriptor that never matches any Atom in the World.
 */
class AtomNoneDescriptor_impl : public AtomDescriptor_impl {
public:
  AtomNoneDescriptor_impl();
  virtual ~AtomNoneDescriptor_impl();

  /**
   * Always returns false for any Atom
   */
  virtual bool predicate(std::pair<atomId_t,atom*>);
};

/************************** Operator stuff ************************/

/**
 * Intersection of two AtomDescriptors
 */
class AtomAndDescriptor_impl : public AtomDescriptor_impl
{
public:
  AtomAndDescriptor_impl(AtomDescriptor::impl_ptr _lhs, AtomDescriptor::impl_ptr _rhs);
  ~AtomAndDescriptor_impl();

  /**
   * This predicate uses the predicate from the first && the predicate from the
   * second Descriptor to decide if an Atom should be selected.
   */
  virtual bool predicate(std::pair<atomId_t,atom*>);

private:
  AtomDescriptor::impl_ptr lhs;
  AtomDescriptor::impl_ptr rhs;
};

/**
 * Union of two AtomDescriptors
 */
class AtomOrDescriptor_impl : public AtomDescriptor_impl
{
public:
  AtomOrDescriptor_impl(AtomDescriptor::impl_ptr _lhs, AtomDescriptor::impl_ptr _rhs);
  virtual ~AtomOrDescriptor_impl();

  /**
   * This predicate uses the predicate form the first || the predicate from the
   * second Descriptor to decide if an Atom should be selected.
   */
  virtual bool predicate(std::pair<atomId_t,atom*>);

private:
  AtomDescriptor::impl_ptr lhs;
  AtomDescriptor::impl_ptr rhs;
};

/**
 * Set Inversion of a Descriptor
 */
class AtomNotDescriptor_impl : public AtomDescriptor_impl
{
public:
  AtomNotDescriptor_impl(AtomDescriptor::impl_ptr _arg);
  virtual ~AtomNotDescriptor_impl();

  /**
   * Opposite of the given descriptor predicate.
   */
  virtual bool predicate(std::pair<atomId_t,atom*>);

private:
  AtomDescriptor::impl_ptr arg;
};

#endif //ATOMDESCRIPTOR_IMPL_HPP
