/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LinkedCell_ModelUnitTest.cpp
 *
 *  Created on: Nov 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <list>

#include "Box.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinkedCell/LinkedCell.hpp"
#include "LinkedCell/LinkedCell_Model.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"

#include "LinkedCell_ModelUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

#define DOMAINLENGTH 20.
#define EDGELENGTH 2.
#define NUMBERCELLS DOMAINLENGTH/EDGELENGTH

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( LinkedCell_ModelTest );


void LinkedCell_ModelTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  setVerbosity(3);

  // create diag(20.) matrix
  RealSpaceMatrix BoxM;
  BoxM.setIdentity();
  BoxM *= DOMAINLENGTH;

  // create Box with this matrix
  domain = new Box(BoxM);

  // create LinkedCell structure with this Box
  LC = new LinkedCell::LinkedCell_Model(EDGELENGTH, *domain);

  // create a list of nodes and add to LCImpl
  std::vector< Vector > VectorList;
  for (size_t i=0;i<((size_t)floor(NUMBERCELLS));++i)
    VectorList.push_back(Vector((double)i*EDGELENGTH,(double)i*EDGELENGTH,(double)i*EDGELENGTH));
  for (size_t i=0;i<VectorList.size();++i) {
    TesselPoint * Walker = new TesselPoint();
    Walker->setName(std::string("Walker")+toString(i));
    Walker->setPosition(VectorList[i]);
    NodeList.insert(Walker);
  }
}


void LinkedCell_ModelTest::tearDown()
{
  delete LC;
  delete domain;

  // remove all nodes again
  for (PointSet::iterator iter = NodeList.begin();
      !NodeList.empty();
      iter = NodeList.begin()) {
    delete *iter;
    NodeList.erase(iter);
  }
}

/** UnitTest for correct construction
 */
void LinkedCell_ModelTest::AllocationTest()
{
  // check that first cell is allocated
  LinkedCell::tripleIndex index;
  index[0] = index[1] = index[2] = 0;
  CPPUNIT_ASSERT(LC->N(index) != NULL);

  // check that very last cell is allocated
  index[0] = index[1] = index[2] = (size_t)floor(NUMBERCELLS)-1;
  CPPUNIT_ASSERT(LC->N(index) != NULL);

}

/** UnitTest for getSize()
 */
void LinkedCell_ModelTest::getSizeTest()
{
  // check getSize()
  for(size_t i=0; i<NDIM; ++i)
    CPPUNIT_ASSERT_EQUAL((LinkedCell::LinkedCellArray::index)floor(NUMBERCELLS), LC->getSize(i));
#ifndef NDEBUG
  std::cout << "The following assertion is intended and is not a failure of the code." << std::endl;
  CPPUNIT_ASSERT_THROW( LC->getSize(4), Assert::AssertionFailure);
#endif
}

/** UnitTest for Reset()
 */
void LinkedCell_ModelTest::ResetTest()
{
  LC->Reset();

  for(size_t i=0; i<NDIM; ++i)
    CPPUNIT_ASSERT_EQUAL((LinkedCell::LinkedCellArray::index)0, LC->getSize(i));
}


/** UnitTest for insertPointCloud()
 */
void LinkedCell_ModelTest::insertPointCloudTest()
{

  // create the linked cell structure
  PointCloudAdaptor< PointSet > cloud(&NodeList, std::string("NodeList"));
  LC->insertPointCloud(cloud);

  // test structure
  CPPUNIT_ASSERT_EQUAL(((size_t)floor(NUMBERCELLS)), LC->CellLookup.size());
  LinkedCell::tripleIndex index;
  for (size_t i=0;i<((size_t)floor(NUMBERCELLS));++i) {
    index[0] = index[1] = index[2] = i;
    // assert that in the destined cell we have one Walker
    CPPUNIT_ASSERT_EQUAL((size_t)1, LC->N(index)->size());
  }
  index[0] = 9;
  index[1] = index[2] = 0;
  // assert that in the destined cell we have one Walker
  CPPUNIT_ASSERT_EQUAL((size_t)0, LC->N(index)->size());

}

/** UnitTest for insertPointCloud()
 */
void LinkedCell_ModelTest::setPartitionTest()
{
  RealSpaceMatrix Pmatrix = LC->Partition;
  RealSpaceMatrix Dmatrix = LC->Dimensions;

  LC->Reset();

  LC->setPartition(2.*EDGELENGTH);

  Pmatrix *= 0.5;
  Dmatrix *= 0.5;

  CPPUNIT_ASSERT_EQUAL(Pmatrix, LC->Partition);
  CPPUNIT_ASSERT_EQUAL(Dmatrix, LC->Dimensions);
}

/** UnitTest for insertPointCloud()
 */
void LinkedCell_ModelTest::getStepTest()
{
  // zero distance
  LinkedCell::tripleIndex index = LC->getStep(0.);
  for (size_t i = 0; i<NDIM; ++i)
    CPPUNIT_ASSERT( (size_t)0 == index[i]);
  // check all possible shells on boundary
  for (double length = EDGELENGTH; length < DOMAINLENGTH; length+=EDGELENGTH) {
    LinkedCell::tripleIndex index = LC->getStep(length);
    for (size_t i = 0; i<NDIM; ++i) {
      std::cout << (size_t)(length/EDGELENGTH) << " ==" << index[i] << std::endl;
      CPPUNIT_ASSERT( (LinkedCell::LinkedCellArray::index)(length/EDGELENGTH) == index[i]);
    }
  }
  // check all possible shells at half interval
  for (double length = 0.5 * EDGELENGTH; length < DOMAINLENGTH; length+=EDGELENGTH) {
    LinkedCell::tripleIndex index = LC->getStep(length);
    for (size_t i = 0; i<NDIM; ++i)
      CPPUNIT_ASSERT( (LinkedCell::LinkedCellArray::index)ceil(length/EDGELENGTH) == index[i]);
  }
}

/** UnitTest for insertPointCloud()
 */
void LinkedCell_ModelTest::getIndexToVectorTest()
{
  {
    const Vector test(0.,0.,0.);
    const LinkedCell::tripleIndex index = LC->getIndexToVector(test);
    for (size_t i = 0; i<NDIM; ++i)
      CPPUNIT_ASSERT( (size_t)0 == index[i]);
  }
  {
    const Vector test(DOMAINLENGTH/2.,DOMAINLENGTH/2.,DOMAINLENGTH/2.);
    const LinkedCell::tripleIndex index = LC->getIndexToVector(test);
    for (size_t i = 0; i<NDIM; ++i)
      CPPUNIT_ASSERT( (size_t)floor(DOMAINLENGTH/EDGELENGTH/2.) == index[i]);
  }
  {
    const Vector test(DOMAINLENGTH/2.,DOMAINLENGTH/3.,DOMAINLENGTH/4.);
    const LinkedCell::tripleIndex index = LC->getIndexToVector(test);
    for (size_t i = 0; i<NDIM; ++i)
      CPPUNIT_ASSERT( (LinkedCell::LinkedCellArray::index)floor(DOMAINLENGTH/EDGELENGTH/(double)(i+2.)) == index[i]);
  }
}

/** UnitTest for insertPointCloud()
 */
void LinkedCell_ModelTest::nodeTest()
{
  // create point
  TesselPoint * Walker = new TesselPoint();
  Walker->setName(std::string("Walker9"));
  Walker->setPosition(Vector(9.8,7.6,5.4));
  PointCloudAdaptor< PointSet > cloud(&NodeList, std::string("NodeList"));
  LC->insertPointCloud(cloud);

  // check addNode
  LC->addNode(Walker);
  CPPUNIT_ASSERT_EQUAL( NodeList.size()+1, LC->CellLookup.size());
  LinkedCell::tripleIndex index1 = LC->getIndexToVector(Walker->getPosition());
  const LinkedCell::tripleIndex &index2 = LC->CellLookup[Walker]->getIndices();
  CPPUNIT_ASSERT(index1 == index2);

  // check moveNode
#ifndef NDEBUG
  std::cout << "The following assertion is intended and is not a failure of the code." << std::endl;
  CPPUNIT_ASSERT_THROW( LC->moveNode(Walker), Assert::AssertionFailure );
#endif

  // check deleteNode
  LC->deleteNode(Walker);
  CPPUNIT_ASSERT_EQUAL( NodeList.size(), LC->CellLookup.size());

  delete Walker;
}
