/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RotateAroundSelfByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "atom.hpp"
#include "molecule.hpp"


#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#include "Actions/MoleculeAction/RotateAroundSelfByAngleAction.hpp"

// and construct the stuff
#include "RotateAroundSelfByAngleAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performCall() {
  // obtain information
  getParametersfromValueStorage();

  // check whether a molecule is selected
  std::vector<molecule *> selectedMolecules = World::getInstance().getSelectedMolecules();
  if (selectedMolecules.size() == 0)
    return Action::failure;

  // go through all selected molecules
  BOOST_FOREACH(molecule *mol, selectedMolecules) {
    // check whether Axis is valid
    if (params.Axis.IsZero())
      return Action::failure;

    // convert from degrees to radian
    params.angle *= M_PI/180.;

    // Creation Line that is the rotation axis
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
    DoLog(0) && (Log() << Verbose(0) <<  "Center of gravity is " << *CenterOfGravity << "." << std::endl);
    Line RotationAxis(*CenterOfGravity, params.Axis);
    delete(CenterOfGravity);
    DoLog(0) && (Log() << Verbose(0) << "Rotate " << mol->getName() << " around self by " << params.angle << " radian around axis " << RotationAxis << "." << endl);

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), params.angle));
    }
    DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  }

  return Action::state_ptr(new MoleculeRotateAroundSelfByAngleState(selectedMolecules, params));
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performUndo(Action::state_ptr _state) {
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  BOOST_FOREACH(molecule *mol, state->selectedMolecules) {
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
    DoLog(0) && (Log() << Verbose(0) <<  "Center of gravity is " << *CenterOfGravity << "." << std::endl);
    Line RotationAxis(*CenterOfGravity, state->params.Axis);
    delete(CenterOfGravity);
    DoLog(0) && (Log() << Verbose(0) << "Rotate " << mol->getName() << " around self by " << -state->params.angle << " radian around axis " << RotationAxis << "." << endl);

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), -state->params.angle));
    }
  }

  return Action::state_ptr(_state);
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performRedo(Action::state_ptr _state){
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  BOOST_FOREACH(molecule *mol, state->selectedMolecules) {
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
    DoLog(0) && (Log() << Verbose(0) <<  "Center of gravity is " << *CenterOfGravity << "." << std::endl);
    Line RotationAxis(*CenterOfGravity, state->params.Axis);
    delete(CenterOfGravity);
    DoLog(0) && (Log() << Verbose(0) << "Rotate " << mol->getName() << " around self by " << state->params.angle << " radian around axis " << RotationAxis << "." << endl);

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), state->params.angle));
    }
  }

  return Action::state_ptr(_state);
}

bool MoleculeRotateAroundSelfByAngleAction::canUndo() {
  return true;
}

bool MoleculeRotateAroundSelfByAngleAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
