/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * TesselationUnitTest.cpp
 *
 *  Created on: Aug 26, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>

#include "CodePatterns/Log.hpp"
#include "Helpers/defs.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "LinkedCell/linkedcell.hpp"
#include "Tesselation/BoundaryLineSet.hpp"
#include "Tesselation/BoundaryTriangleSet.hpp"
#include "Tesselation/CandidateForTesselation.hpp"
#include "Atom/TesselPoint.hpp"

#include "TesselationUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

const double TesselationTest::SPHERERADIUS=2.;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( TesselationTest );


void TesselationTest::setUp()
{
  // create corners
  class TesselPoint *Walker;
  Walker = new TesselPoint;
  Walker->setPosition(Vector(1., 0., -1.));
  Walker->setName("1");
  Walker->setNr(1);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(-1., 1., -1.));
  Walker->setName("2");
  Walker->setNr(2);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(-1., -1., -1.));
  Walker->setName("3");
  Walker->setNr(3);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(-1., 0., 1.));
  Walker->setName("4");
  Walker->setNr(4);
  Corners.push_back(Walker);

  // create tesselation
  TesselStruct = new Tesselation;
  CPPUNIT_ASSERT_EQUAL( true, TesselStruct->PointsOnBoundary.empty() );
  CPPUNIT_ASSERT_EQUAL( true, TesselStruct->LinesOnBoundary.empty() );
  CPPUNIT_ASSERT_EQUAL( true, TesselStruct->TrianglesOnBoundary.empty() );
  PointCloudAdaptor<TesselPointSTLList> cloud(&Corners, "TesselPointSTLList");
  (*TesselStruct)(cloud, SPHERERADIUS);
};


void TesselationTest::tearDown()
{
  delete(TesselStruct);
  for (TesselPointSTLList::iterator Runner = Corners.begin(); Runner != Corners.end(); Runner++)
    delete(*Runner);
  Corners.clear();
  logger::purgeInstance();
  errorLogger::purgeInstance();
};

/** UnitTest for Contains...()
 *
 */
void TesselationTest::ContainmentTest()
{
  class BoundaryPointSet *point = NULL;
  class BoundaryLineSet *line = NULL;

  // check ContainsBoundaryPoint
  for(LineMap::iterator Runner = TesselStruct->LinesOnBoundary.begin(); Runner != TesselStruct->LinesOnBoundary.end(); Runner++) {
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryPoint((Runner->second)->endpoints[0]));
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryPoint((Runner->second)->endpoints[1]));
  }
  for(TriangleMap::iterator Runner = TesselStruct->TrianglesOnBoundary.begin(); Runner != TesselStruct->TrianglesOnBoundary.end(); Runner++) {
    for(PointMap::iterator PointRunner = TesselStruct->PointsOnBoundary.begin(); PointRunner != TesselStruct->PointsOnBoundary.end(); PointRunner++) {
      point = PointRunner->second;
      for (int i=0;i<3;i++)
        if (point == (Runner->second)->endpoints[i])
          point = NULL;
      if (point != NULL)
        break;
    }
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryPoint((Runner->second)->endpoints[0]));
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryPoint((Runner->second)->endpoints[1]));
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryPoint((Runner->second)->endpoints[2]));
    CPPUNIT_ASSERT_EQUAL( false, (Runner->second)->ContainsBoundaryPoint(point));
  }

  // check ContainsBoundaryLine
  for(TriangleMap::iterator Runner = TesselStruct->TrianglesOnBoundary.begin(); Runner != TesselStruct->TrianglesOnBoundary.end(); Runner++) {
    for(LineMap::iterator LineRunner = TesselStruct->LinesOnBoundary.begin(); LineRunner != TesselStruct->LinesOnBoundary.end(); LineRunner++) {
      line = LineRunner->second;
      for (int i=0;i<3;i++)
        if (line == (Runner->second)->lines[i])
          line = NULL;
      if (line != NULL)
        break;
    }
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryLine((Runner->second)->lines[0]));
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryLine((Runner->second)->lines[1]));
    CPPUNIT_ASSERT_EQUAL( true, (Runner->second)->ContainsBoundaryLine((Runner->second)->lines[2]));
    CPPUNIT_ASSERT_EQUAL( false, (Runner->second)->ContainsBoundaryLine(line));
  }

  // check IsPresentTupel
  CPPUNIT_ASSERT_EQUAL( true, true );
}

/** UnitTest for Tesselation::GetAllTriangles()
 *
 */
void TesselationTest::GetAllTrianglesTest()
{
  class BoundaryPointSet *Walker = NULL;

  // check that there are three adjacent triangles for every boundary point
  for (PointMap::iterator Runner = TesselStruct->PointsOnBoundary.begin(); Runner != TesselStruct->PointsOnBoundary.end(); Runner++) {
    Walker = Runner->second;
    set<BoundaryTriangleSet*> *triangles = TesselStruct->GetAllTriangles(Walker);
    CPPUNIT_ASSERT_EQUAL( (size_t)3, triangles->size() );
    // check that the returned triangle all contain the Walker
    for (set<BoundaryTriangleSet*>::iterator TriangleRunner = triangles->begin(); TriangleRunner != triangles->end(); TriangleRunner++)
      CPPUNIT_ASSERT_EQUAL( true, (*TriangleRunner)->ContainsBoundaryPoint(Walker) );
    delete(triangles);
  }
}
