/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CreateMicelleAction.cpp
 *
 *  Created on: Sept 29, 2010
 *      Author: dueck
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "CodePatterns/MemDebug.hpp"

#include "Actions/ActionHistory.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/GraphAction/SubgraphDissectionAction.hpp"
#include "Actions/MoleculeAction/RotateToPrincipalAxisSystemAction.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"

#include "atom.hpp"
#include "Bond/bond.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <string>
#include <vector>

#include "Parser/PdbParser.hpp"
#include "Parser/TremoloParser.hpp"
#include "Parser/XyzParser.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/ShapeOps.hpp"

#include "Actions/MoleculeAction/CreateMicelleAction.hpp"

#include "CreateMicelleAction.def"
#include "Action_impl_pre.hpp"

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"


/** =========== define the function ====================== */
Action::state_ptr MoleculeCreateMicelleAction::performCall()
{
  typedef std::vector <atom *> AtomVector;
  typedef std::vector <molecule *> MoleculeVector;

  getParametersfromValueStorage();
	
  AtomVector ever = World::getInstance().getAllAtoms();

  // as all parsed atoms go into same molecule
  // we don't need to create one and add them all to it
  MoleculeVector all = World::getInstance().getSelectedMolecules();
  ASSERT(!all.empty(), "MoleculeCreateMicelleAction::performCall() - no molecules selected.");
  molecule *stick = *(all.begin());

  // center molecule

  stick->CenterOrigin();

  // determine principal axis system and rotate such that greatest extension is along z axis
  Vector den(0.0,0.0,1.0);

  MoleculeRotateToPrincipalAxisSystem(den);
  /* determine
  principal axis system and make greatest eigenvector be aligned along
  (0,0,1)
  */
  string path;

  Line RotationAxis(Vector(0.0,0.0,0.0), Vector(1.0,0.0,0.0)); // pt is the current Vector of point on surface

  //for (molecule::iterator it=stick->begin(); it !=stick->end(); ++it)
    //(*it)->setPosition(RotationAxis.rotateVector((*it)->getPosition(),M_PI));

  /// Align molecule with its PAS multiple times with the some surface

  // get points on surface

  int ka =0;
  //double radius= 1.5*sqrt(pow(1.55, 2)*params.N);

  Shape s = resize(Sphere(), params.radius);
  std::vector<Vector> pt = s.getHomogeneousPointsOnSurface(params.N);

  // mirror along x-y plane

  for (molecule::iterator it2=stick->begin();it2 !=stick->end();++it2)
  {
	  Vector pos= (**it2).getPosition();
	  pos[2]= - pos[2];// -Min[2]
	  (**it2).setPosition(pos);
  }

  // shift molecule by its extension along z axis

  for (molecule::iterator it2=stick->begin();it2 !=stick->end();++it2)
  {
	  Vector pos= (**it2).getPosition();
	  pos[2]=pos[2]+params.radius; // -Min[2]
	  (**it2).setPosition(pos);
  }

  // copy molecule N times and rotate it to point radially away from surface

  //double MYEPSILON=1e-10;

  for (ka = 0; ka<pt.size()-1; ka++)
  {
    LOG(1, "STATUS: Creating " << ka+1 << " copy of tenside molecule 'stick' with " << stick->getAtomCount() << " atoms, ");
	  molecule *Tensid=stick->CopyMolecule();

	  LOG(2, "INFO: Rotating ... ");
    Vector ZAxis(Vector(0.0,0.0,1.0));
    Vector Axis(pt[ka]);
    const double alpha = ZAxis.Angle(Axis);
    Axis.VectorProduct(ZAxis);
    Line RotationAxis(Vector(0.0,0.0,1.0), Axis);	// pt is the current Vector of point on surface

    for (molecule::iterator it2=Tensid->begin();it2 !=Tensid->end();++it2)
    {
      (*it2)->setPosition(RotationAxis.rotateVector((*it2)->getPosition(),alpha));
      *(*it2)+=params.center;
    }
    LOG(2, "INFO: \tdone.");

    Tensid=NULL;
  }

  // shift molecule at given position on surface

  LOG(1, "STATUS: Shifting " << ka+1 << " copy of tenside molecule, ");
  molecule *Tensid=stick;
  LOG(2, "INFO: Rotating ...");
  Vector ZAxis(Vector(0.0,0.0,1.0));
  Vector Axis(pt[pt.size()-1]);
  const double alpha = ZAxis.Angle(Axis);
  Axis.VectorProduct(ZAxis);
  Line RotationAxis2(Vector(0.0,0.0,1.0), Axis);	// pt is the current Vector of point on surface

  for (molecule::iterator it2=Tensid->begin();it2 !=Tensid->end();++it2)
  {
    (*it2)->setPosition(RotationAxis.rotateVector((*it2)->getPosition(),alpha));
    *(*it2)+=params.center;
  }
  LOG(2, "INFO: \tdone.");

  Tensid=NULL;

  return Action::success;
}

Action::state_ptr MoleculeCreateMicelleAction::performUndo(Action::state_ptr _state)
{
  //  MoleculeFillVoidWithMoleculeState *state = assert_cast<MoleculeFillVoidWithMoleculeState*>(_state.());

  //  string newName = state->mol->Name();
  //  state->mol->setName(state->lastName);

  return Action::failure;
}

Action::state_ptr MoleculeCreateMicelleAction::performRedo(Action::state_ptr _state)
{
  // Undo and redo have to do the same for this action
  return performUndo(_state);
}


bool MoleculeCreateMicelleAction::canUndo()
{
  return false;
}

bool MoleculeCreateMicelleAction::shouldUndo()
{
  return false;
}

/** =========== end of function ====================== */
