/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * builder_init.cpp
 *
 *  Created on: Dec 15, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost::python uses placement new which is incompatible with MemDebug.
#ifdef HAVE_PYTHON
#include "Python/PythonScripting.hpp"
#endif

//#include "CodePatterns/MemDebug.hpp"

#include <boost/filesystem.hpp>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <iterator>
#ifdef HAVE_PTHREAD
#include "ThreadGuard.hpp"
#endif
#include <string>

#include "Actions/ActionExceptions.hpp"

#include "builder_init.hpp"
#include "cleanUp.hpp"

#include "CodePatterns/Log.hpp"

//#include "Graph/BondGraph.hpp"

#include "UIElements/UIFactory.hpp"
#include "UIElements/CommandLineUI/CommandLineParser.hpp"
#include "UIElements/CommandLineUI/CommandLineUIFactory.hpp"
#ifdef USE_GUI_QT
#include "UIElements/Qt4/QtUIFactory.hpp"
#endif
#include "UIElements/TextUI/TextUIFactory.hpp"
#include "UIElements/MainWindow.hpp"

#include "version.h"

#include <boost/filesystem.hpp>

/** Print some initial information output the program.
 *
 */
void ProgramHeader()
{
  // print version check and copyright notice
  cout << MOLECUILDERVERSION << endl;
  cout << "MoleCuilder comes with ABSOLUTELY NO WARRANTY; for details type" << endl;
  cout << "`molecuilder --warranty'." << endl;
  cout << "MoleCuilder - to create and alter molecular systems." << endl;
  cout << "Copyright (C) 2008-2012  University Bonn. All rights reserved." << endl;
  cout << "Copyright (C) 2013-2016  Frederik Heber. All rights reserved." << endl;
  cout << "This is free software, and you are welcome to redistribute it under" << endl;
  cout << "certain conditions; type `molecuilder --help-redistribute' for details" << endl;
  cout << endl;
}

#ifdef HAVE_PTHREAD
pthread_t actionqueue_global_threadid = 0;
#endif

/** General stuff to intialize before UI.
 *
 */
void initGeneral()
{
  // while we are non interactive, we want to abort from asserts
  ASSERT_DO(Assert::Abort);
  ASSERT_HOOK(dumpMemory);

  ProgramHeader();

  setVerbosity(1);

#ifdef HAVE_PTHREAD
  actionqueue_global_threadid = pthread_self();
#endif

  // from this moment on, we need to be sure to deeinitialize in the correct order
  // this is handled by the cleanup function
  atexit(cleanUp);
}

/** Initialize specific UIFactory.
 *
 * @param argc argument count
 * @param argv argument array
 */
void initUI(int argc, char **argv)
{
  //std::string BondGraphFileName("\n");
  // Parse command line options and if present create respective UI
  // construct bond graph
  //if (boost::filesystem::exists(BondGraphFileName)) {
  //  std::ifstream input(BondGraphFileName.c_str());
  //  if ((input.good()) && (World::getInstance().getBondGraph()->LoadBondLengthTable(input))) {
  //    LOG(0, "Bond length table loaded successfully.");
  //  } else {
  //    ELOG(1, "Bond length table loading failed.");
  //  }
  //  input.close();
  //}

  // handle remaining arguments by CommandLineParser
  // first argument of "--test" initializes special molecuilder gui test mode
  if ((argc>1)
#ifdef USE_GUI_QT
      && !QtUIFactory::isTestMode(argv[1])
      && !QtUIFactory::isGuiMode(argv[1])
#endif
  ) {
    LOG(0, "Setting UI to CommandLine.");
    CommandLineParser::getInstance().InitializeCommandArguments();
    CommandLineParser::getInstance().Run(argc,argv);
    UIFactory::registerFactory(new CommandLineUIFactory::description());
    UIFactory::makeUserInterface("CommandLine");
  } else {
    // In the interactive mode, we can leave the user the choice in case of error
    #ifdef USE_GUI_QT
    if ((argc>1) && (QtUIFactory::isTestMode(argv[1])))
      ASSERT_DO(Assert::Abort);
    else
      ASSERT_DO(Assert::Ask);
      LOG(0, "Setting UI to Qt4.");
      UIFactory::registerFactory(new QtUIFactory::description(argc, argv));
      UIFactory::makeUserInterface("Qt4");
    #else
      ASSERT_DO(Assert::Ask);
      LOG(0, "Setting UI to Text.");
      cout << MOLECUILDERVERSION << endl;
      UIFactory::registerFactory(new TextUIFactory::description());
      UIFactory::makeUserInterface("Text");
    #endif
  }
}

/** Create MainWindow and displays.
 * I.e. here all the Actions are parsed and done.
 */
void doUI()
{
  MainWindow *mainWindow = UIFactory::getInstance().makeMainWindow();

  // wait till is done initializing
  while (!UIFactory::isDoneInitializing());

  LOG(0, "UI is done initializing.");

  // if we have python, autoexecute a molecuilder script in current folder
#ifdef HAVE_PYTHON
  const boost::filesystem::path pythonfilename("./molecuilder.py");
  executePythonScriptFile(pythonfilename);
#endif

  mainWindow->display();

  delete mainWindow;
}

