/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * StretchBondAction.cpp
 *
 *  Created on: Sep 26, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Actions/MoleculeAction/StretchBondAction.hpp"

#include <boost/bind.hpp>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include "LinearAlgebra/Plane.hpp"

#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Graph/BoostGraphCreator.hpp"
#include "Graph/BoostGraphHelpers.hpp"
#include "Graph/BreadthFirstSearchGatherer.hpp"
#include "molecule.hpp"
#include "World.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "StretchBondAction.def"
#include "Action_impl_pre.hpp"


static bool addEdgePredicate(
    const bond &_bond,
    const std::vector<atomId_t> &_atomids)
{
  ASSERT(_atomids.size() == (size_t)2,
      "addEdgePredicate() - atomids must contain exactly two ids.");
  // do not add selected edge
  return ((_bond.leftatom->getId() != _atomids[0])
      || (_bond.rightatom->getId() != _atomids[1]));
}

/** =========== define the function ====================== */
ActionState::ptr MoleculeStretchBondAction::performCall()
{
  // check preconditions
  const std::vector< atom *> atoms = World::getInstance().getSelectedAtoms();
  if (atoms.size() != 2) {
    STATUS("Exactly two atoms must be selected.");
    return Action::failure;
  }
  molecule *mol = World::getInstance().
      getMolecule(MoleculeById(atoms[0]->getMolecule()->getId()));
  if (mol != atoms[1]->getMolecule()) {
    STATUS("The two selected atoms must belong to the same molecule.");
    return Action::failure;
  }

  // gather undo information
  const double olddistance = atoms[0]->getPosition().distance(atoms[1]->getPosition());
  const double newdistance = params.bonddistance.get();
  LOG(1, "INFO: Old bond distance is " << olddistance << ", stretching to " << newdistance << ".");

  // gather sorted ids
  std::vector<atomId_t> atomids(2);
  atomids[0] = atoms[0]->getId();
  atomids[1] = atoms[1]->getId();
  std::sort(atomids.begin(), atomids.end());
  LOG(1, "DEBUG: Selected nodes are " << atomids);

  const Vector NormalVector = (atoms[0]->getPosition() - atoms[1]->getPosition())* (1./olddistance);
  const double shift = 0.5*(newdistance - olddistance);
  std::vector<Vector> Shift(2);
  Shift[0] = shift * NormalVector;
  Shift[1] = -shift * NormalVector;
  Box &domain = World::getInstance().getDomain();

  // Assume the selected bond splits the molecule into two parts, each one on
  // either side of the bond. We need to perform a BFS from each bond partner
  // not using the selected bond. Therefrom, we obtain two sets of atoms/nodes.
  // If both are disjoint, the bond is not contained in a cycle and we simply
  // shift either set as desired. If not, then we simply shift each atom,
  // leaving the other positions untouched.

  // get nodes on either side of selected bond via BFS discovery
  BoostGraphCreator BGcreator;
  BGcreator.createFromMolecule(*mol,
      boost::bind(addEdgePredicate, _1, boost::ref(atomids)));
  BreadthFirstSearchGatherer NodeGatherer(BGcreator);
  std::vector< BoostGraphHelpers::Nodeset_t > bondside_sets(2);
  for(size_t j=0;j<2;++j) {
    bondside_sets[j] = NodeGatherer(atoms[j]->getId());
    std::sort(bondside_sets[j].begin(), bondside_sets[j].end());
  }

  // simple test whether bond has split the system in two disjoint sets or not
  bool isCyclic = false;
  if ((bondside_sets[0].size() + bondside_sets[1].size()) > BGcreator.getNumVertices()) {
      // Check whether there are common nodes in each set of distances
    if (BoostGraphHelpers::isCommonNodeInVector(bondside_sets[0], bondside_sets[1])) {
      ELOG(2, "Sets contain common node, hence bond must have been by cyclic."
          << " Shifting only bond partners.");
      for(size_t j=0;j<2;++j) {
        bondside_sets[j].clear();
        bondside_sets[j].push_back( atomids[j] );
        const Vector &position = atoms[j]->getPosition();
        atoms[j]->setPosition( domain.enforceBoundaryConditions(position+Shift[j]) );
      }
      isCyclic = true;
    }
  }

  // go through the molecule and stretch each atom in either set of nodes
  if (!isCyclic) {
    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      const Vector &position = (*iter)->getPosition();
      // for each atom determine in which set of nodes it is and shift accordingly
      const atomId_t &atomid = (*iter)->getId();
      if (std::binary_search(bondside_sets[0].begin(), bondside_sets[0].end(), atomid)) {
        (*iter)->setPosition( domain.enforceBoundaryConditions(position+Shift[0]) );
      } else if (std::binary_search(bondside_sets[1].begin(), bondside_sets[1].end(), atomid)) {
        (*iter)->setPosition( domain.enforceBoundaryConditions(position+Shift[1]) );
      } else {
        ELOG(1, "Atom " << *iter << " is not contained on either side of bond? Undoing done shifts");
        // Have to undo shifts
        for (size_t i=0;i<2;++i) {
          for (BoostGraphHelpers::Nodeset_t::const_iterator iter = bondside_sets[i].begin();
              iter != bondside_sets[i].end(); ++iter) {
            atom &walker = *World::getInstance().getAtom(AtomById(*iter));
            const Vector &position = walker.getPosition();
            walker.setPosition( domain.enforceBoundaryConditions(position-Shift[i]) );
          }
        }
        return Action::failure;
      }
    }
  }

  MoleculeStretchBondState *UndoState = new MoleculeStretchBondState(Shift, bondside_sets, mol, params);
  return ActionState::ptr(UndoState);
}

ActionState::ptr MoleculeStretchBondAction::performUndo(ActionState::ptr _state) {
  MoleculeStretchBondState *state = assert_cast<MoleculeStretchBondState*>(_state.get());

  // use given plane to undo
  Box &domain = World::getInstance().getDomain();
  for (size_t i=0;i<2;++i) {
    for (BoostGraphHelpers::Nodeset_t::const_iterator iter = state->bondside_sets[i].begin();
        iter != state->bondside_sets[i].end(); ++iter) {
      atom &walker = *World::getInstance().getAtom(AtomById(*iter));
      const Vector &position = walker.getPosition();
      walker.setPosition( domain.enforceBoundaryConditions(position-state->Shift[i]) );
    }
  }

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeStretchBondAction::performRedo(ActionState::ptr _state){
  MoleculeStretchBondState *state = assert_cast<MoleculeStretchBondState*>(_state.get());

  Box &domain = World::getInstance().getDomain();
  for (size_t i=0;i<2;++i) {
    for (BoostGraphHelpers::Nodeset_t::const_iterator iter = state->bondside_sets[i].begin();
        iter != state->bondside_sets[i].end(); ++iter) {
      atom &walker = *World::getInstance().getAtom(AtomById(*iter));
      const Vector &position = walker.getPosition();
      walker.setPosition( domain.enforceBoundaryConditions(position+state->Shift[i]) );
    }
  }

  return ActionState::ptr(_state);
}

bool MoleculeStretchBondAction::canUndo() {
  return true;
}

bool MoleculeStretchBondAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
