/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserMpqcUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserMpqcUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Parser/MpqcParser.hpp"
#include "World.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserMpqcUnitTest );

static string waterMpqc ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<MBPT2>: (\n\
\t\tbasis = $:basis\n\
\t\tmolecule = $:molecule\n\
\t\treference<CLHF>: (\n\
\t\t\tmaxiter = 1000\n\
\t\t\tbasis = $:basis\n\
\t\t\tmolecule = $:molecule\n\
\t\t)\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n";

void ParserMpqcUnitTest::setUp() {
  World::getInstance();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserMpqcUnitTest::tearDown() {
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserMpqcUnitTest::writeMpqcTest() {
  // build up water molecule
  atom *Walker = NULL;
  Walker = World::getInstance().createAtom();
  Walker->setType(8);
  Walker->setPosition(Vector(0,0,0));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,0.504284));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,-0.504284));
  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  // create two stringstreams, one stored, one created
  stringstream input(waterMpqc);
  MpqcParser* testParser = new MpqcParser();
  stringstream output;
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  testParser->save(&output, atoms);

  // compare both configs
  string first = input.str();
  string second = output.str();
  CPPUNIT_ASSERT(first == second);
}
