/*
 * SaturatedBond.hpp
 *
 *  Created on: Jul 27, 2014
 *      Author: heber
 */

#ifndef SATURATEDBOND_HPP_
#define SATURATEDBOND_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>
#include <iosfwd>
#include <boost/shared_ptr.hpp>

#include "LinearAlgebra/Vector.hpp"

#include "Bond/bond.hpp"

class SaturationDistanceMaximizer;
class SaturatedBond;

std::ostream &operator<<(std::ostream &ost, const SaturatedBond& _bond);

/** This class contains a dangling bond and information how to saturate it
 * with hydrogens.
 *
 * Hydrogens are place on a cylinder on the sphere around the remaining atom.
 * The angle \a alpha describes the position along the circle cutting the
 * sphere. The symmetry axis of the cylinder is the bond direction. The
 * orthogonal vectors \a vector_a and \a vector_b describe the circle on the
 * sphere.
 */
class SaturatedBond
{
  //!> allow output operator const access to private parts
  friend std::ostream &operator<<(std::ostream &ost, const SaturatedBond& _bond);

public:
  //!> typedef for a shared pointer of this class
  typedef boost::shared_ptr<SaturatedBond> ptr;

  /** Cstor of class SaturatedBond, requires a present \a _bond.
   *
   * \param _bond bond to saturate
   * \param _remaining atom that is not cut off
   */
  SaturatedBond(const bond &_bond, const atom& _remaining);

  /** Dstor of SaturatedBond.
   *
   */
  ~SaturatedBond() {}

  //!> typedef for a vector of Vectors
  typedef std::vector<Vector> positions_t;

  /** calculates positions where to place hydrogens to saturate this bond.
   *
   * \note Positions are always relative to \a saturated, i.e. to the atom
   * that is saturated.
   */
  positions_t getPositions() const;

private:
  //!> SaturationDistanceMinimizer needs access to alpha and directional vectors
  friend class SaturationDistanceMaximizer;

  //!> bond to saturate
  const bond& saturated_bond;
  //!> central atom which is saturated
  const atom& saturated_atom;
  //!> rotation angle describing position.
  double alpha;
  //!> Vector along the bond
  Vector BondVector;
  //!> first vector orthogonal to \a _bond
  Vector vector_a;
  //!> second vector orthogonal to \a _bond
  Vector vector_b;
};



#endif /* SATURATEDBOND_HPP_ */
