/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * UndoRedoHelpers.cpp
 *
 *  Created on: Apr 5, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "UndoRedoHelpers.hpp"

#include <boost/foreach.hpp>

#include "Atom/atom.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "World.hpp"

bool MoleCuilder::AddAtomsFromAtomicInfo(std::vector<AtomicInfo> &atoms)
{
  size_t i=0;
  for (; i<atoms.size(); ++i) {
    // re-create the atom
    LOG(3, "DEBUG: Re-adding atom " << atoms[i].getId() << ".");
    atom *Walker = World::getInstance().createAtom();
    if (!atoms[i].setAtom(*Walker)) {
      ELOG(1, "Failed to set id.");
      World::getInstance().destroyAtom(Walker);
      break;
    }
  }
  if (i<atoms.size()) {
    // remove all previous ones, too
    for (size_t j=0;j<i;++j)
      World::getInstance().destroyAtom(atoms[j].getId());
    // and announce the failure
    return false;
  }
  return true;
}

void MoleCuilder::RemoveAtomsFromAtomicInfo(std::vector<AtomicInfo> &atoms)
{
  BOOST_FOREACH(const AtomicInfo &_atom, atoms) {
    World::getInstance().destroyAtom(_atom.getId());
  }
}

void MoleCuilder::StoreBondInformationFromAtoms(
    const std::vector<const atom*> &atoms,
    std::vector< BondInfo > &bonds)
{
  ASSERT( bonds.empty(),
      "StoreBondInformationFromAtoms() - give bonds vector is not empty.");
  bonds.reserve(atoms.size()*4);
  for (std::vector<const atom*>::const_iterator atomiter = atoms.begin();
      atomiter != atoms.end(); ++atomiter) {
    const BondList & _atom_bonds = (*atomiter)->getListOfBonds();
    for(BondList::const_iterator iter = _atom_bonds.begin(); iter != _atom_bonds.end(); ++iter)
      bonds.push_back( BondInfo(*iter) );
  }
}

bool MoleCuilder::AddBondsFromBondInfo(const std::vector< BondInfo > &bonds)
{
  bool status = true;
  for(std::vector< BondInfo >::const_iterator iter = bonds.begin();
      iter != bonds.end(); ++iter)
    if (!(*iter).RecreateBond())
      status = false;
  return status;
}

void MoleCuilder::SetAtomsFromAtomicInfo(std::vector<AtomicInfo> &movedatoms)
{
  BOOST_FOREACH( const AtomicInfo &_atominfo, movedatoms) {
    const atomId_t id = _atominfo.getId();
    atom * const _atom = World::getInstance().getAtom(AtomById(id));
    ASSERT( _atom != NULL,
        "MoleCuilder::SetAtomsFromAtomicInfo() - cannot find atom with id "
        +toString(id)+" in the world.");
    _atominfo.setAtom( *_atom );
  }
}

void MoleCuilder::SelectAtomsFromAtomicInfo(std::vector<AtomicInfo> &movedatoms)
{
  BOOST_FOREACH( const AtomicInfo &_atominfo, movedatoms) {
    const atomId_t id = _atominfo.getId();
    World::getInstance().selectAtom(id);
  }
}

void MoleCuilder::ResetAtomPosition(const std::vector<AtomicInfo> &movedatoms, const std::vector<Vector> &MovedToVector)
{
  std::vector<Vector>::const_iterator positer = MovedToVector.begin();
  ASSERT(movedatoms.size() == MovedToVector.size(),
      "MoleCuilder::ResetAtomPosition() -  the number of atoms "
      +toString(movedatoms.size())+" and the number of positions "
      +toString(MovedToVector.size())+" is not the same.");
  BOOST_FOREACH( const AtomicInfo &_atominfo, movedatoms) {
    const atomId_t id = _atominfo.getId();
    atom * const _atom = World::getInstance().getAtom(AtomById(id));
    ASSERT( _atom != NULL,
        "FillSphericalSurfaceAction::performRedo() - cannot find atom with id "
        +toString(id)+" in the world.");
    _atom->setPosition( *positer );
    ++positer;
  }
}
