/*
 * MemDebug.hpp
 *
 *  Created on: Apr 28, 2010
 *      Author: crueger
 */

#ifndef MEMDEBUG_HPP_
#define MEMDEBUG_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/**
 * @file
 * This module allows easy automatic memory tracking. Link this module to replace the
 * operators new, new[], delete and delete[] with custom versions that add tracking
 * information to allocated blocks.
 *
 * All tracking is done in O(1) for new and delete operators. Summaries for the
 * used memory take O(N), where N is the number of currently allocated memory chunks.
 *
 * To use full tracking including file name and line number include this file in
 * your sourcefiles.
 */

// Set all flags in a way that makes sense

// NDEBUG implies NO_MEMDEBUG
#ifdef NDEBUG
# ifndef NO_MEMDEBUG
#   define NO_MEMDEBUG
# endif
#endif

// NO_MEMDEBUG and MEMDEBUG are mutually exclusive, but at least one must be set
#ifdef NO_MEMDEBUG
# ifdef MEMDEBUG
#   undef MEMDEBUG
# endif
#else
# ifndef MEMDEBUG
#   define MEMDEBUG
# endif
#endif

#ifdef MEMDEBUG

#include <new>

// some light header files, that do weird new stuff and therefore need
// to be loaded before the define
#include <string>
#if defined HAVE_BOOST_OPTIONAL_HPP || defined HAVE_BOOST_THREAD_HPP
#include <boost/optional.hpp>
#endif
#if defined HAVE_BOOST_SHARED_PTR_HPP || defined HAVE_BOOST_THREAD_HPP
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#endif
#ifdef HAVE_BOOST_BIND_HPP
#include <boost/bind.hpp>
#endif
#if defined HAVE_BOOST_FUNCTION_HPP || defined HAVE_BOOST_THREAD_HPP
#include <boost/function.hpp>
#endif
#ifdef HAVE_BOOST_ARCHIVE_TEXT_OARCHIVE_HPP
// serialization has bug with overloaded new, see https://svn.boost.org/trac/boost/ticket/3400
#include <boost/serialization/access.hpp>
#include <boost/archive/detail/iserializer.hpp>
#endif
#if defined HAVE_VALARRAY || defined HAVE_BOOST_ARCHIVE_TEXT_OARCHIVE_HPP
// valarray uses specific new as well
#include <valarray>
#endif
#if defined HAVE_BOOST_MULTI_ARRAY_HPP || defined HAVE_BOOST_ASIO_HPP
// multiarray uses type trait has_new_operator.hpp
#include <boost/type_traits/has_new_operator.hpp>
#endif
#if defined HAVE_BOOST_ASIO_HPP
// boost asio uses own allocator
#include <boost/serialization/shared_ptr_132.hpp>
#endif
#if defined HAVE_BOOST_MPL_FOR_EACH_HPP
// boost mpl requires the value_init which uses placement new
#include <boost/utility/value_init.hpp>
#endif
#if defined HAVE_BOOST_EXCEPTION_ALL_HPP
// boost exceptions has two inaccessible static objects for bad_alloc and bad_exception
#include <boost/exception/detail/exception_ptr.hpp>
#endif

namespace Memory {

  /**
   * Displays a short summary of the memory state.
   */
  void getState();
  void dumpMemory(std::ostream&);

  void _ignore(void*);

  /**
   * Use this to disable memory for a certain pointer on the heap.
   * This is useful for pointers which should live over the run of the
   * program, and which are deleted automatically at the end. Usually these
   * pointers should be wrapped inside some kind of smart pointer to
   * ensure destruction at the end.
   */
  template <typename T>
  T *ignore(T* ptr){
    _ignore((void*)ptr);
    return ptr;
  }
}
#ifdef __GNUC__
void *operator new   (size_t nbytes,const char* file, int line, const char* func) throw(std::bad_alloc);
void *operator new[] (size_t nbytes,const char* file, int line, const char* func) throw(std::bad_alloc);
#else
void *operator new   (size_t nbytes,const char* file, int line) throw(std::bad_alloc);
void *operator new[] (size_t nbytes,const char* file, int line) throw(std::bad_alloc);
#endif
void operator delete   (void *ptr,const char*, int) throw();
void operator delete[] (void *ptr,const char*, int) throw();



/**
 * This macro replaces all occurences of the keyword new with a replaced
 * version that allows tracking.
 */
#ifdef __GNUC__
#define new new(__FILE__,__LINE__,__PRETTY_FUNCTION__)
#else
#define new new(__FILE__,__LINE__)
#endif

#else

#include <iosfwd>

// memory debugging was disabled

namespace Memory {
  inline void getState(){};

  inline void dumpMemory(std::ostream&){};

  template <typename T>
  inline T *ignore(T* ptr){
    return ptr;
  }
}

#endif


#endif /* MEMDEBUG_HPP_ */
