/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserXyzUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserXyzUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Parser/XyzParser.hpp"
#include "World.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserXyzUnitTest );

static string waterXyz = "3\n\tH2O: water molecule\nO\t0\t0\t0\nH\t0.758602\t0\t0.504284\nH\t0.758602\t0\t-0.504284\n";

void ParserXyzUnitTest::setUp() {
  World::getInstance();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserXyzUnitTest::tearDown() {
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserXyzUnitTest::rewriteAnXyzTest() {
  cout << "Testing the XYZ parser." << endl;
  XyzParser* testParser = new XyzParser();
  stringstream input;
  input << waterXyz;
  testParser->load(&input);
  input.clear();

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  // store and parse in again
  {
    stringstream output;
    std::vector<atom *> atoms = World::getInstance().getAllAtoms();
    testParser->save(&output, atoms);
    input << output.str();
    testParser->load(&input);
  }

  // now twice as many
  CPPUNIT_ASSERT_EQUAL(6, World::getInstance().numAtoms());

  // check every atom
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  std::vector<atom *>::const_iterator firstiter = atoms.begin();
  std::vector<atom *>::const_iterator seconditer = atoms.begin();
  for (size_t i=0;i<3;i++)
    ++seconditer;
  for (;
      seconditer != atoms.end();
      ++firstiter,++seconditer) {
    // check position and type (only stuff xyz stores)
    CPPUNIT_ASSERT_EQUAL((*firstiter)->getPosition(),(*seconditer)->getPosition());
    CPPUNIT_ASSERT_EQUAL((*firstiter)->getType(),(*seconditer)->getType());
  }
}
