/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QTDialog.cpp
 *
 *  Created on: Jan 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "UIElements/QT4/QTDialog.hpp"

#include <boost/lexical_cast.hpp>

#include <string>
#include <sstream>
#include <limits>

#include <Qt/qboxlayout.h>
#include <Qt/qlabel.h>
#include <Qt/qspinbox.h>
#include <QtGui/QDoubleSpinBox>
#include <Qt/qlineedit.h>
#include <Qt/qlistwidget.h>
#include <Qt/qdialogbuttonbox.h>
#include <Qt/qpushbutton.h>
#include <Qt/qcombobox.h>

#include <boost/lexical_cast.hpp>

#include "Helpers/MemDebug.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "Box.hpp"


using namespace std;

QTDialog::QTDialog() :
    QDialog(0)
{
  // creating and filling of the Dialog window
  mainLayout = new QVBoxLayout();
  inputLayout = new QVBoxLayout();
  buttonLayout = new QVBoxLayout();
  setLayout(mainLayout);
  mainLayout->addLayout(inputLayout);
  mainLayout->addLayout(buttonLayout);
  buttons = new QDialogButtonBox(QDialogButtonBox::Ok| QDialogButtonBox::Cancel);
  buttonLayout->addWidget(buttons);

  // Disable the ok button until something was entered
  buttons->button(QDialogButtonBox::Ok)->setEnabled(false);

  // connect the buttons to their appropriate slots
  connect(buttons, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttons, SIGNAL(rejected()), this, SLOT(reject()));
}

QTDialog::~QTDialog()
{
}

bool QTDialog::display(){
  // Button state might have changed by some update that
  // was done during query construction. To make sure
  // the state is correct, we just call update one more time.
  update();
  if(exec()) {
    setAll();
    return true;
  }
  else {
    return false;
  }
}

void QTDialog::update(){
  buttons->button(QDialogButtonBox::Ok)->setEnabled(checkAll());
}

/************************** Query Infrastructure ************************/

void QTDialog::queryEmpty(const char* title, std::string)
{
  registerQuery(new EmptyQTQuery(title,inputLayout,this));
}

void QTDialog::queryBoolean(const char*,string){
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryAtom(const char*, std::string){
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryAtoms(const char*, std::string){
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryBox(const char*, std::string){
  // TODO
  ASSERT(false, "Not implemented yet");
}


void QTDialog::queryInt(const char *title,string)
{
  registerQuery(new IntQTQuery(title,inputLayout,this));
}

void QTDialog::queryInts(const char *title,string)
{
  registerQuery(new IntsQTQuery(title,inputLayout,this));
}

void QTDialog::queryDouble(const char* title,string)
{
  registerQuery(new DoubleQTQuery(title,inputLayout,this));
}

void QTDialog::queryDoubles(const char* title,string)
{
  registerQuery(new DoublesQTQuery(title,inputLayout,this));
}

void QTDialog::queryString(const char* title,string)
{
  registerQuery(new StringQTQuery(title,inputLayout,this));
}

void QTDialog::queryStrings(const char* title,string)
{
  registerQuery(new StringsQTQuery(title,inputLayout,this));
}

void QTDialog::queryMolecule(const char *title,string)
{
  registerQuery(new MoleculeQTQuery(title,inputLayout,this));
}

void QTDialog::queryMolecules(const char *title,string)
{
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryVector(const char* title, bool check,string)
{
  registerQuery(new VectorQTQuery(title,check,inputLayout,this));
}

void QTDialog::queryVectors(const char* title, bool check,string)
{
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryElement(const char* title, std::string)
{
  registerQuery(new ElementQTQuery(title,inputLayout,this));
}

void QTDialog::queryElements(const char* title, std::string)
{
  // TODO
  ASSERT(false, "Not implemented yet");
}

void QTDialog::queryFile(const char* title, std::string)
{
  registerQuery(new FileQTQuery(title,inputLayout,this));
}

/************************** Query Objects *******************************/

QTDialog::IntQTQuery::IntQTQuery(string _title,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::IntQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QSpinBox();
  inputBox->setValue(0);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new IntQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::IntQTQuery::~IntQTQuery()
{
  delete pipe;
}

bool QTDialog::IntQTQuery::handle() {
  return true;
}


QTDialog::IntsQTQuery::IntsQTQuery(string _title,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::IntsQuery(_title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new QTQueryListPipe<int>(&tmp,_dialog,inputBox,inputList,AddButton,RemoveButton);
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(IntegerEntered(const QString&)));
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(AddValue()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(RemoveRow()));
}

QTDialog::IntsQTQuery::~IntsQTQuery()
{
  delete pipe;
}

bool QTDialog::IntsQTQuery::handle() {
  return true;
}


QTDialog::DoubleQTQuery::DoubleQTQuery(string title,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::DoubleQuery(title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QDoubleSpinBox();
  inputBox->setValue(0);
  inputBox->setRange(-numeric_limits<double>::max(),numeric_limits<double>::max());
  inputBox->setDecimals(3);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new DoubleQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
}

QTDialog::DoubleQTQuery::~DoubleQTQuery()
{
  delete pipe;
}

bool QTDialog::DoubleQTQuery::handle() {
  return true;
}


QTDialog::DoublesQTQuery::DoublesQTQuery(string title,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::DoublesQuery(title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new QTQueryListPipe<double>(&tmp,_dialog,inputBox,inputList,AddButton,RemoveButton);
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(IntegerEntered(const QString&)));
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(AddValue()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(RemoveRow()));}

QTDialog::DoublesQTQuery::~DoublesQTQuery()
{
  delete pipe;
}

bool QTDialog::DoublesQTQuery::handle() {
  return true;
}


QTDialog::StringQTQuery::StringQTQuery(string _title,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::StringQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QLineEdit();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new StringQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->text());
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(update(const QString&)));
}

QTDialog::StringQTQuery::~StringQTQuery()
{
  delete pipe;
}

// All values besides the empty std::string are valid
bool QTDialog::StringQTQuery::handle()
{
  return tmp!="";
}

QTDialog::StringsQTQuery::StringsQTQuery(string _title,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::StringsQuery(_title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new QTQueryListPipe<std::string>(&tmp,_dialog,inputBox,inputList,AddButton,RemoveButton);
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(IntegerEntered(const QString&)));
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(AddValue()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(RemoveRow()));}

QTDialog::StringsQTQuery::~StringsQTQuery()
{
  delete pipe;
}

// All values besides the empty std::string are valid
bool QTDialog::StringsQTQuery::handle()
{
  // dissect by ","
  std::string::iterator olditer = temp.begin();
  for(string::iterator iter = temp.begin(); iter != temp.end(); ++iter) {
    if (*iter == ' ') {
      tmp.push_back(string(iter, olditer));
      olditer = iter;
    }
  }
  if (olditer != temp.begin())  // insert last part also
    tmp.push_back(string(olditer, temp.end()));

  return temp!="";
}

QTDialog::MoleculeQTQuery::MoleculeQTQuery(string _title, QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::MoleculeQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  // add all molecules to the combo box
  vector<molecule*> molecules = World::getInstance().getAllMolecules();
  for(vector<molecule*>::iterator iter  = molecules.begin();
      iter != molecules.end();
      ++iter) {
    std::stringstream sstr;
    sstr << (*iter)->IndexNr << "\t" << (*iter)->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter)->IndexNr));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new MoleculeQTQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::MoleculeQTQuery::~MoleculeQTQuery()
{
  delete pipe;
}

// Handling is easy, since the GUI makes it impossible to select invalid values
bool QTDialog::MoleculeQTQuery::handle()
{
  return true;
}

QTDialog::MoleculesQTQuery::MoleculesQTQuery(string _title, QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::MoleculesQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  // add all molecules to the combo box
  vector<molecule*> molecules = World::getInstance().getAllMolecules();
  for(vector<molecule*>::iterator iter  = molecules.begin();
      iter != molecules.end();
      ++iter) {
    std::stringstream sstr;
    sstr << (*iter)->IndexNr << "\t" << (*iter)->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter)->IndexNr));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new MoleculesQTQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::MoleculesQTQuery::~MoleculesQTQuery()
{
  delete pipe;
}

// Handling is easy, since the GUI makes it impossible to select invalid values
bool QTDialog::MoleculesQTQuery::handle()
{
  return true;
}

QTDialog::VectorQTQuery::VectorQTQuery(std::string title, bool _check,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::VectorQuery(title,_check),
    parent(_parent)
{
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QVBoxLayout();
  mainLayout->addLayout(subLayout);
  QComboBox* inputBox = new QComboBox();
  coordLayout = new QHBoxLayout();
  subLayout->addLayout(coordLayout);
  coordLabel = new QLabel(QString("x,y,z"));
  coordLayout->addWidget(coordLabel);
  coordInput = new QDoubleSpinBox();
//  coordInput->setRange(0,M.at(i,i));
  coordInput->setDecimals(3);
  coordLayout->addWidget(coordInput);
  pipe = new VectorQTQueryPipe(&(tmp),_dialog,inputBox);
  //pipe->update(coordInput->value());
  connect(coordInput,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
  parent->addLayout(mainLayout);
}

QTDialog::VectorQTQuery::~VectorQTQuery()
{}

bool QTDialog::VectorQTQuery::handle() {
  return true;
}


QTDialog::VectorsQTQuery::VectorsQTQuery(std::string title, bool _check,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::VectorsQuery(title,_check),
    parent(_parent)
{
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QVBoxLayout();
  mainLayout->addLayout(subLayout);
  QComboBox* inputBox = new QComboBox();
  coordLayout = new QHBoxLayout();
  subLayout->addLayout(coordLayout);
  coordLabel = new QLabel(QString("x,y,z"));
  coordLayout->addWidget(coordLabel);
  coordInput = new QDoubleSpinBox();
//  coordInput->setRange(0,M.at(i,i));
  coordInput->setDecimals(3);
  coordLayout->addWidget(coordInput);
  pipe = new VectorsQTQueryPipe(&(tmp),_dialog,inputBox);
  //pipe->update(coordInput->value());
  connect(coordInput,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
  parent->addLayout(mainLayout);
}

QTDialog::VectorsQTQuery::~VectorsQTQuery()
{}

bool QTDialog::VectorsQTQuery::handle() {
  return true;
}


QTDialog::ElementQTQuery::ElementQTQuery(std::string _title, QBoxLayout *_parent, QTDialog *_dialog) :
    Dialog::ElementQuery(_title),
    parent(_parent)
{
  periodentafel *periode = World::getInstance().getPeriode();
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  for(periodentafel::const_iterator iter = periode->begin();
      iter!=periode->end();
      ++iter)
  {
    std::stringstream sstr;
    sstr << (*iter).first << "\t" << (*iter).second->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter).first));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new ElementQTQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::ElementQTQuery::~ElementQTQuery()
{
  delete pipe;
}

bool QTDialog::ElementQTQuery::handle(){
  return true;
}


QTDialog::ElementsQTQuery::ElementsQTQuery(std::string _title, QBoxLayout *_parent, QTDialog *_dialog) :
    Dialog::ElementsQuery(_title),
    parent(_parent)
{
  periodentafel *periode = World::getInstance().getPeriode();
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  for(periodentafel::const_iterator iter = periode->begin();
      iter!=periode->end();
      ++iter)
  {
    std::stringstream sstr;
    sstr << (*iter).first << "\t" << (*iter).second->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter).first));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new ElementsQTQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::ElementsQTQuery::~ElementsQTQuery()
{
  delete pipe;
}

bool QTDialog::ElementsQTQuery::handle(){
  return true;
}

QTDialog::EmptyQTQuery::EmptyQTQuery(std::string _title, QBoxLayout *_parent, QTDialog *_dialog) :
    Dialog::EmptyQuery(_title),
    parent(_parent)
{
  titleLabel = new QLabel(QString(getTitle().c_str()));

  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);

  pipe = new EmptyQTQueryPipe(_dialog,titleLabel);
}

QTDialog::EmptyQTQuery::~EmptyQTQuery()
{
  delete pipe;
}

bool QTDialog::EmptyQTQuery::handle(){
  return true;
}


QTDialog::FileQTQuery::FileQTQuery(std::string _title, QBoxLayout *_parent, QTDialog *_dialog) :
    Dialog::FileQuery(_title),
    parent(_parent)
{

  filenameLineEdit = new QLineEdit(_dialog);
  filenameLineEdit->setText(QString());
  filenameLineEdit->setReadOnly(true);

  filenameLabel = new QLabel(QString("Input file:"));
  filenameLabel->setBuddy(filenameLineEdit);

  filedialogButton = new QPushButton("&Choose", _dialog);

  pipe = new FileQTQueryPipe(&tmp,_dialog,filenameLineEdit,filedialogButton);

  thisLayout = new QHBoxLayout();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(filenameLabel);
  thisLayout->addWidget(filenameLineEdit);
  thisLayout->addWidget(filedialogButton);

  QObject::connect(filedialogButton,SIGNAL(clicked()),pipe,SLOT(showFileDialog()));
}

QTDialog::FileQTQuery::~FileQTQuery()
{
  delete pipe;
}

bool QTDialog::FileQTQuery::handle(){
  return true;
}

/*************************** Plumbing *******************************/


template<typename T> QTQueryListPipe<T>::QTQueryListPipe(std::vector<T> *_content, QTDialog *_dialog, QLineEdit *_inputBox, QListWidget *_inputList, QPushButton *_AddButton, QPushButton *_RemoveButton) :
  content(_content),
  dialog(_dialog),
  inputBox(_inputBox),
  inputList(_inputList),
  AddButton(_AddButton),
  RemoveButton(_RemoveButton)
{}

template<typename T> QTQueryListPipe<T>::~QTQueryListPipe()
{}

template<typename T> void QTQueryListPipe<T>::IntegerEntered(const QString&)
{
  AddButton->setEnabled(true);
}

template<typename T> void QTQueryListPipe<T>::IntegerSelected()
{
  if (inputList->selectedItems().empty())
    RemoveButton->setEnabled(false);
  else
    RemoveButton->setEnabled(true);
}

template<typename T> void QTQueryListPipe<T>::AddInteger() {
  // type-check
  std::string text = inputBox->text().toStdString();
  int number = 0;
  try {
    number = boost::lexical_cast<int>(text);
  } catch (boost::bad_lexical_cast&) {
    return;
  };
  // add item to both
  inputList->addItem(QString(number));
  AddValue(number);
}

template<typename T> void QTQueryListPipe<T>::AddValue(T item) {
  content->push_back(item);

  dialog->update();
}

template<typename T> void QTQueryListPipe<T>::RemoveInteger() {
  QList<QListWidgetItem *> items = inputList->selectedItems();
  for (QList<QListWidgetItem *>::iterator iter = items.begin(); !items.empty(); iter = items.begin()) {
    // obtain which position item has (by making it current item)
    inputList->setCurrentItem(*iter);
    // remove
    QTQueryListPipe<T>::RemoteRow(inputList->currentRow()); // template parameters needs to be known, such that compiler knows which to call
    inputList->removeItemWidget(*iter);
  }
}

template<typename T> void QTQueryListPipe<T>::RemoveRow(int row) {
  int counter = 0;
  typename std::vector<T>::iterator iter = content->begin();
  for (; iter != content->end(); ++iter)
    if (counter++ == row)
      break;
  if (iter != content->end())
      content->erase(iter);
}


StringQTQueryPipe::StringQTQueryPipe(string *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

StringQTQueryPipe::~StringQTQueryPipe()
{}

void StringQTQueryPipe::update(const QString& newText) {
  content->assign(newText.toStdString());
  dialog->update();
}

IntQTQueryPipe::IntQTQueryPipe(int *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

IntQTQueryPipe::~IntQTQueryPipe()
{}

void IntQTQueryPipe::update(int newInt) {
  (*content) = newInt;
  dialog->update();
}

DoubleQTQueryPipe::DoubleQTQueryPipe(double *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

DoubleQTQueryPipe::~DoubleQTQueryPipe()
{}

void DoubleQTQueryPipe::update(double newDbl) {
  (*content) = newDbl;
  dialog->update();
}

VectorQTQueryPipe::VectorQTQueryPipe(Vector *_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

VectorQTQueryPipe::~VectorQTQueryPipe()
{}

void VectorQTQueryPipe::update() {
  dialog->update();
}

VectorsQTQueryPipe::VectorsQTQueryPipe(std::vector<Vector> *_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

VectorsQTQueryPipe::~VectorsQTQueryPipe()
{}

void VectorsQTQueryPipe::update() {
  dialog->update();
}

AtomQTQueryPipe::AtomQTQueryPipe(atom **_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

AtomQTQueryPipe::~AtomQTQueryPipe()
{}

void AtomQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = World::getInstance().getAtom(AtomById(idx));
  dialog->update();
}


AtomsQTQueryPipe::AtomsQTQueryPipe(std::vector<atom *>*_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

AtomsQTQueryPipe::~AtomsQTQueryPipe()
{}

void AtomsQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  atom *Walker = World::getInstance().getAtom(AtomById(idx));
  if (Walker)
    (*content).push_back(Walker) ;
  dialog->update();
}


MoleculeQTQueryPipe::MoleculeQTQueryPipe(molecule **_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

MoleculeQTQueryPipe::~MoleculeQTQueryPipe()
{}

void MoleculeQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = World::getInstance().getMolecule(MoleculeById(idx));
  dialog->update();
}


MoleculesQTQueryPipe::MoleculesQTQueryPipe(std::vector<molecule *>*_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

MoleculesQTQueryPipe::~MoleculesQTQueryPipe()
{}

void MoleculesQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  molecule *mol = World::getInstance().getMolecule(MoleculeById(idx));
  if (mol)
    (*content).push_back(mol);
  dialog->update();
}

ElementQTQueryPipe::ElementQTQueryPipe(const element **_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

ElementQTQueryPipe::~ElementQTQueryPipe()
{}

void ElementQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  *content = World::getInstance().getPeriode()->FindElement(idx);
  dialog->update();
}

ElementsQTQueryPipe::ElementsQTQueryPipe(std::vector<const element *>*_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

ElementsQTQueryPipe::~ElementsQTQueryPipe()
{}

void ElementsQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  const element *elemental = World::getInstance().getPeriode()->FindElement(idx);
  if(elemental)
    (*content).push_back(elemental);
  dialog->update();
}

EmptyQTQueryPipe::EmptyQTQueryPipe(QTDialog *_dialog, QLabel *_textLabel) :
  dialog(_dialog),
  textLabel(_textLabel)
{}

EmptyQTQueryPipe::~EmptyQTQueryPipe()
{}

void EmptyQTQueryPipe::update() {
  dialog->update();
}

FileQTQueryPipe::FileQTQueryPipe(boost::filesystem::path *_content, QTDialog *_dialog, QLineEdit *_filenameLineEdit, QPushButton *_filedialogButton) :
  content(_content),
  dialog(_dialog),
  filenameLineEdit(_filenameLineEdit),
  filedialogButton(_filedialogButton)
{
  theFileDialog = NULL;
}

FileQTQueryPipe::~FileQTQueryPipe()
{
  if (theFileDialog != NULL)
    delete theFileDialog;
}

void FileQTQueryPipe::update() {
  QStringList ListOfFilenames = theFileDialog->selectedFiles();
  std::cout << "Selected File is " << ListOfFilenames.at(0).toStdString() << std::endl;
  (*content) = ListOfFilenames.at(0).toStdString();
  filenameLineEdit->setText(QString::fromStdString((*content).string()));
  dialog->update();
}

void FileQTQueryPipe::showFileDialog() {
  filedialogButton->setFlat(true);
  if (theFileDialog == NULL) {
    theFileDialog = new QFileDialog(NULL, tr("Open input file"), QString(), tr("ParallelCarParrinello (*.conf);;MassivelyParallelQuantumChemistry (*.mpqc);;ParticleDataBase (*.pdb);;XYZ (*.xyz)"));
    theFileDialog->setAcceptMode(QFileDialog::AcceptOpen);
    theFileDialog->setFileMode(QFileDialog::ExistingFile);
    theFileDialog->setViewMode(QFileDialog::List);
  }
  theFileDialog->exec();

  update();
  filedialogButton->setFlat(false);
}



