/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserMpqcUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserMpqcUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Parser/MpqcParser.hpp"
#include "World.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserMpqcUnitTest );

static string waterMpqc_CLHF ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<CLHF>: (\n\
\t\tmolecule = $:molecule\n\
\t\tbasis = $:basis\n\
\t\tmemory = 16000000\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n"; // tested with mpqc 3.0.0-alpha
static string waterMpqc_CLKS ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<CLKS>: (\n\
\t\tfunctional<StdDenFunctional>:(name=B3LYP)\n\
\t\tmolecule = $:molecule\n\
\t\tbasis = $:basis\n\
\t\tmemory = 16000000\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n"; // tested with mpqc 3.0.0-alpha
static string waterMpqc_MBPT2 ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<MBPT2>: (\n\
\t\tbasis = $:basis\n\
\t\tmolecule = $:molecule\n\
\t\treference<CLHF>: (\n\
\t\t\tmaxiter = 1000\n\
\t\t\tbasis = $:basis\n\
\t\t\tmolecule = $:molecule\n\
\t\t)\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n"; // tested with mpqc 3.0.0-alpha
static string waterMpqc_MBPT2_R12 ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<MBPT2_R12>: (\n\
\t\tmolecule = $:molecule\n\
\t\tbasis = $:basis\n\
\t\taux_basis = $:abasis\n\
\t\tstdapprox = \"A'\"\n\
\t\tnfzc = 1\n\
\t\tmemory = 16000000\n\
\t\tintegrals<IntegralCints>:()\n\
\t\treference<CLHF>: (\n\
\t\t\tmolecule = $:molecule\n\
\t\t\tbasis = $:basis\n\
\t\t\tmaxiter = 1000\n\
\t\t\tmemory = 16000000\n\
\t\t\tintegrals<IntegralCints>:()\n\
\t\t)\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n\
% auxiliary basis set specification\n\
\tabasis<GaussianBasisSet>: (\n\
\tname = \"aug-cc-pVDZ\"\n\
\tmolecule = $:molecule\n\
)\n"; // basically tested with mpqc 3.0.0-alpha (no parse errors but did not calculate due to missing code)

void ParserMpqcUnitTest::setUp() {
  World::getInstance();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserMpqcUnitTest::tearDown() {
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserMpqcUnitTest::readMpqcTest() {
  stringstream input(waterMpqc_CLHF);
  MpqcParser* testParser = new MpqcParser();
  testParser->setTheory(MpqcParser::CLHF);
  testParser->load(&input);

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
}

void ParserMpqcUnitTest::writeMpqcTest() {
  // build up water molecule
  string first;
  string second;
  atom *Walker = NULL;
  Walker = World::getInstance().createAtom();
  Walker->setType(8);
  Walker->setPosition(Vector(0,0,0));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,0.504284));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,-0.504284));
  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  // create two stringstreams, one stored, one created

  MpqcParser* testParser = new MpqcParser();
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  {
    // compare both configs for CLHF
    stringstream output;
    testParser->setTheory(MpqcParser::CLHF);
    testParser->save(&output, atoms);
    stringstream input(waterMpqc_CLHF);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
  {
    // compare both configs for CLKS
    stringstream output;
    testParser->setTheory(MpqcParser::CLKS);
    testParser->save(&output, atoms);
    stringstream input(waterMpqc_CLKS);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
  {
    // compare both configs for MBPT2
    stringstream output;
    testParser->setTheory(MpqcParser::MBPT2);
    testParser->save(&output, atoms);
    stringstream input(waterMpqc_MBPT2);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
  {
    // compare both configs for MBPT2_R12
    stringstream output;
    testParser->setTheory(MpqcParser::MBPT2_R12);
    testParser->save(&output, atoms);
    stringstream input(waterMpqc_MBPT2_R12);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
}
