/*
 * Observer.hpp
 *
 *  Created on: Jan 19, 2010
 *      Author: crueger
 */

#ifndef OBSERVER_HPP_
#define OBSERVER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <set>
#include <string>
#include <sstream>

// Deactivate any logging when we are not in debug mode
#ifdef NDEBUG
#undef LOG_OBSERVER
#endif

#include "CodePatterns/Observer/defs.hpp"

class Channels;
class Observable;

/**
 * An Observer is notified by all Observed objects, when anything changes.
 *
 * If a simple change is done to an Object the Obervable will call the update() method
 * of all signed on observers, passing itself as a parameter for identification. The
 * Observers should then react to the changes and update themselves accordingly.
 *
 * If an observed Object is destroyed it will call the subjectKilled() method
 * of all signed on Observers, again passing itself as a parameter for identification.
 * The Observers should handle the destruction of an observed Object gracefully, i.e.
 * set themselves inactive, display something else, etc. There is no need
 * to sign of from the dying object, since this will be handled by the Observable destructor.
 */
class Observer
{
  friend class Notification;
  friend class Observable;
  friend class Relay;
  template<class> friend class ObservedIterator;

  // indicates the constructor called from Observables
  struct BaseConstructor{};

public:
  Observer(BaseConstructor);
  Observer(std::string);
  virtual ~Observer();

protected:
  /**
   * This method is called upon changes of the Observable
   */
  virtual void update(Observable *publisher)=0;

  /**
   * This method is called when a special named change
   * of the Observable occured
   */
  virtual void recieveNotification(Observable *publisher, Notification_ptr notification);

  /**
   * This method is called when the observed object is destroyed.
   */
  virtual void subjectKilled(Observable *publisher)=0;
};

#ifdef LOG_OBSERVER
#include "ObserverLog.hpp"
#endif


#endif /* OBSERVER_HPP_ */
