/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * NoseHoover.cpp
 *
 *  Created on: Aug 20, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "NoseHoover.hpp"

#include "CodePatterns/Log.hpp"
#include "config.hpp"
#include "Element/element.hpp"
#include "Helpers/defs.hpp"
#include "Parser/PcpParser_helper.hpp"
#include "Thermostats/ThermoStatContainer.hpp"
#include "World.hpp"

NoseHoover::NoseHoover(double _HooverMass) :
  HooverMass(_HooverMass),
  alpha(0)
{}

NoseHoover::NoseHoover() :
  HooverMass(0.)
{}

NoseHoover::~NoseHoover()
{}

const char *ThermostatTraits<NoseHoover>::name = "NoseHoover";

std::string ThermostatTraits<NoseHoover>::getName(){
  return ThermostatTraits<NoseHoover>::name;
}

Thermostat *ThermostatTraits<NoseHoover>::make(class ConfigFileBuffer * const fb){
  double HooverMass;
  const int verbose = 0;
  ParseForParameter(verbose,fb,"Thermostat", 0, 2, 1, double_type, &HooverMass, 1, critical); // read Hoovermass
  return new NoseHoover(HooverMass);
}

double NoseHoover::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::list) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double NoseHoover::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::vector) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double NoseHoover::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::set) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double NoseHoover::doScaleAtoms(unsigned int step,double ActualTemp,ForwardIterator begin, ForwardIterator end){
  LOG(2,  "Applying Nose-Hoover thermostat...");
  init(step,begin,end);
  delta_alpha = (delta_alpha - (3.*count+1.) * getContainer().TargetTemp)/(HooverMass*Units2Electronmass);
  alpha += delta_alpha*World::getInstance().getConfig()->Deltat;
  LOG(3, "alpha = " << delta_alpha << " * " << World::getInstance().getConfig()->Deltat << " = " << alpha << ".");
  double ekin =0;
  for(ForwardIterator iter=begin;iter!=end;++iter){
    Vector U = (*iter)->getAtomicVelocityAtStep(step);
    if ((*iter)->getFixedIon() == 0) { // even FixedIon moves, only not by other's forces
      U += World::getInstance().getConfig()->Deltat/(*iter)->getType()->getMass() * (alpha * (U * (*iter)->getType()->getMass()));
      ekin += (0.5*(*iter)->getType()->getMass()) * U.NormSquared();
    }
    (*iter)->setAtomicVelocityAtStep(step, U);
  }
  return ekin;
}

template <class ForwardIterator>
void NoseHoover::init(unsigned int step,ForwardIterator begin, ForwardIterator end){
  delta_alpha=0;
  count=0;
  for(ForwardIterator iter = begin;iter!=end;++iter){
    const Vector &U = (*iter)->getAtomicVelocityAtStep(step);
    if ((*iter)->getFixedIon() == 0) { // even FixedIon moves, only not by other's forces
      delta_alpha += U.NormSquared()*(*iter)->getType()->getMass();
    }
    ++count;
  }
}

std::string NoseHoover::name(){
  return ThermostatTraits<NoseHoover>::name;
}

std::string NoseHoover::writeParams(){
  std::stringstream sstr;
  sstr << HooverMass;
  return sstr.str();
}
