/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Berendsen.cpp
 *
 *  Created on: Aug 20, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Berendsen.hpp"

#include "CodePatterns/Log.hpp"
#include "config.hpp"
#include "Element/element.hpp"
#include "Helpers/defs.hpp"
#include "Parser/PcpParser_helper.hpp"
#include "Thermostats/ThermoStatContainer.hpp"
#include "World.hpp"

Berendsen::Berendsen(double _TempFrequency) :
  TempFrequency(_TempFrequency)
{}

Berendsen::Berendsen() :
  TempFrequency(2.5)
{}

Berendsen::~Berendsen()
{}

const char *ThermostatTraits<Berendsen>::name = "Berendsen";

std::string ThermostatTraits<Berendsen>::getName(){
  return ThermostatTraits<Berendsen>::name;
}

Thermostat *ThermostatTraits<Berendsen>::make(class ConfigFileBuffer * const fb){
  double TempFrequency;
  const int verbose = 0;
  ParseForParameter(verbose,fb,"Thermostat", 0, 2, 1, double_type, &TempFrequency, 1, critical); // read \tau_T
  return new Berendsen(TempFrequency);
}

double Berendsen::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::list) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Berendsen::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::vector) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Berendsen::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::set) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double Berendsen::doScaleAtoms(unsigned int step,double ActualTemp,ForwardIterator begin, ForwardIterator end){
  LOG(2,  "Applying Berendsen-VanGunsteren thermostat...");
  double ekin=0;
  double ScaleTempFactor = getContainer().TargetTemp/ActualTemp;
  for(ForwardIterator iter=begin;iter!=end;++iter){
    Vector U = (*iter)->getAtomicVelocityAtStep(step);
    if ((*iter)->getFixedIon() == 0) { // even FixedIon moves, only not by other's forces
      U *= sqrt(1+(World::getInstance().getConfig()->Deltat/TempFrequency)*(ScaleTempFactor-1));
      ekin += 0.5*(*iter)->getType()->getMass() * U.NormSquared();
    }
    (*iter)->setAtomicVelocityAtStep(step, U);
  }
  return ekin;
}

std::string Berendsen::name(){
  return ThermostatTraits<Berendsen>::name;
}

std::string Berendsen::writeParams(){
  stringstream sstr;
  sstr << TempFrequency;
  return sstr.str();
}
