/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IsVoidNode_FillPredicateUnitTest.cpp
 *
 *  Created on: Jan 18, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <algorithm>
#include <boost/bind.hpp>
#include <iostream>

#include "CodePatterns/Assert.hpp"

#include "Atom/atom.hpp"
#include "Box.hpp"
#include "Filling/Predicates/IsVoidNode_FillPredicate.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "Filling/NodeTypes.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "World.hpp"

#include "IsVoidNode_FillPredicateUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IsVoidNode_FillPredicateTest );

void DoSomething(atom * const _atom)
{
  _atom->setPosition( Vector((double)_atom->getId(), 0., 0.) );
}

void IsVoidNode_FillPredicateTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // set default BCs to ignore
  World::getInstance().getDomain().setConditions(
      BoundaryConditions::Conditions_t(3, BoundaryConditions::Ignore)
  );

  // create some atoms as "neighbours"
  atoms.resize((size_t)5, NULL);
  std::generate_n(atoms.begin(), (size_t)5, boost::bind(&World::createAtom, World::getPointer()) );

  // position them
  std::for_each(atoms.begin(), atoms.end(), &DoSomething);

  predicate = new FillPredicate(IsVoidNode_FillPredicate(Sphere()));
}


void IsVoidNode_FillPredicateTest::tearDown()
{
  delete predicate;

  World::purgeInstance();
}

/** Test whether operator() returns as desired
 *
 */
void IsVoidNode_FillPredicateTest::operatorTest()
{
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( (*predicate)( Vector(-2.,0.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( (*predicate)( Vector(-2.,0.,0.)) );
#endif
  for (double i = 0; i < 5.; ++i) {
    CPPUNIT_ASSERT( !(*predicate)( Vector(i, 0., 0.)) );
    CPPUNIT_ASSERT( !(*predicate)( Vector(i, 1., 0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( (*predicate)( Vector(i, -1., 0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !(*predicate)( Vector(i, -1., 0.)) );
#endif
    CPPUNIT_ASSERT( !(*predicate)( Vector(i, 0., 1.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( (*predicate)( Vector(i, 0., -1.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( !(*predicate)( Vector(i, 0., -1.)) );
#endif
  }
  CPPUNIT_ASSERT( (*predicate)( Vector(5.,1.,0.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(  (*predicate)( Vector(5.,-1.,0.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( (*predicate)( Vector(5.,-1.,0.)) );
#endif
  CPPUNIT_ASSERT( (*predicate)( Vector(5.,0.,1.)) );
#ifndef NDEBUG
  std::cout << "The following Assertion is intended and does not represent a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW( (*predicate)( Vector(5.,0.,-1.)), Assert::AssertionFailure );
#else
  CPPUNIT_ASSERT( (*predicate)( Vector(5.,0.,-1.)) );
#endif
  CPPUNIT_ASSERT( (*predicate)( Vector(6.,0.,0.)) );
}
