/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IsValidInDomain_FillPredicateUnitTest.cpp
 *
 *  Created on: Jan 18, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "CodePatterns/Assert.hpp"

#include "Box.hpp"
#include "Filling/Predicates/IsValidInDomain_FillPredicate.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "Filling/NodeTypes.hpp"

#include "IsValidInDomain_FillPredicateUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IsValidInDomain_FillPredicateTest );


void IsValidInDomain_FillPredicateTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  M = new RealSpaceMatrix;
  M->setIdentity();
  (*M) *= 20.;

  domain = new Box(*M);

  predicate = new FillPredicate(IsValidInDomain_FillPredicate(*domain));
}


void IsValidInDomain_FillPredicateTest::tearDown()
{
  delete predicate;
  delete domain;
  delete M;
}

/** Test whether operator() returns as desired
 *
 */
void IsValidInDomain_FillPredicateTest::operatorTest()
{
  Node origin(0.,0.,0.);
  Node center(10.,10.,10.);
  Node corner(20.,20.,20.);
  Node outside(30.,30.,30.);

  for (size_t i=0;i<NDIM;++i)
    domain->setCondition(i, BoundaryConditions::Wrap);

  // boundary conditions: Wrap
  {
    // origin is inside
    CPPUNIT_ASSERT( (*predicate)(origin) );

    // center is inside
    CPPUNIT_ASSERT( (*predicate)(center) );

    // corner is inside
    CPPUNIT_ASSERT( (*predicate)(corner) );

    // outside is outside
    CPPUNIT_ASSERT( (*predicate)(outside) );
  }

  // boundary conditions: Bounce
  {
    domain->setCondition(0, BoundaryConditions::Bounce);
    // origin is inside
    CPPUNIT_ASSERT( (*predicate)(origin) );

    // center is inside
    CPPUNIT_ASSERT( (*predicate)(center) );

    // corner is inside
    CPPUNIT_ASSERT( (*predicate)(corner) );

    // outside is outside
    CPPUNIT_ASSERT( (*predicate)(outside) );
    domain->setCondition(0, BoundaryConditions::Wrap);
  }

  // boundary conditions: Ignore
  {
    domain->setCondition(0, BoundaryConditions::Ignore);
    // origin is inside
    CPPUNIT_ASSERT( (*predicate)(origin) );

    // center is inside
    CPPUNIT_ASSERT( (*predicate)(center) );

    // corner is inside
    CPPUNIT_ASSERT( (*predicate)(corner) );

    // outside is outside
    CPPUNIT_ASSERT( !(*predicate)(outside) );
    domain->setCondition(0, BoundaryConditions::Wrap);
  }
}
