/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IsInsideSurface_FillPredicateUnitTest.cpp
 *
 *  Created on: Jan 18, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Atom/TesselPoint.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "Filling/Predicates/IsInsideSurface_FillPredicate.hpp"
#include "Filling/NodeTypes.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Tesselation/BoundaryLineSet.hpp"
#include "Tesselation/BoundaryTriangleSet.hpp"
#include "Tesselation/CandidateForTesselation.hpp"
#include "Tesselation/tesselation.hpp"

#include "IsInsideSurface_FillPredicateUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

const double IsInsideSurface_FillPredicateTest::SPHERERADIUS=2.;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IsInsideSurface_FillPredicateTest );


void IsInsideSurface_FillPredicateTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // create corners
  class TesselPoint *Walker;
  Walker = new TesselPoint;
  Walker->setPosition(Vector(1., 0., -1.));
  Walker->setName("1");
  Walker->setNr(1);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(-1., 1., -1.));
  Walker->setName("2");
  Walker->setNr(2);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(-1., -1., -1.));
  Walker->setName("3");
  Walker->setNr(3);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(-1., 0., 1.));
  Walker->setName("4");
  Walker->setNr(4);
  Corners.push_back(Walker);

  // create linkedcell
  PointCloudAdaptor<TesselPointSTLList> cloud(&Corners, "TesselPointSTLList");
  LinkedList = new LinkedCell_deprecated(cloud, 2.*SPHERERADIUS);

  // create tesselation
  TesselStruct = new Tesselation;
  (*TesselStruct)(cloud, SPHERERADIUS);

  // create predicate
  predicate = new FillPredicate(IsInsideSurface_FillPredicate(*TesselStruct, *LinkedList));
}


void IsInsideSurface_FillPredicateTest::tearDown()
{
  delete predicate;

  delete(LinkedList);
  delete(TesselStruct);
  for (TesselPointSTLList::iterator Runner = Corners.begin(); Runner != Corners.end(); Runner++)
    delete(*Runner);
  Corners.clear();
  logger::purgeInstance();
  errorLogger::purgeInstance();

}

/** Test whether operator() returns as desired
 *
 */
void IsInsideSurface_FillPredicateTest::operatorTest()
{
  Node center(0.,0.,0.);
  Node corner(1.,0.,-1.);
  Node outside(2.,0.,0.);

  {
    // center is inside
    CPPUNIT_ASSERT( (*predicate)(center) );
  }

  {
    // corner is inside
    CPPUNIT_ASSERT( (*predicate)(corner) );
  }

  {
    // outside is outside
    CPPUNIT_ASSERT( !(*predicate)(outside) );
  }
}
