/** \file molecule.hpp
 *
 * Class definitions of atom and molecule, element and periodentafel
 */

#ifndef MOLECULES_HPP_
#define MOLECULES_HPP_

/*********************************************** includes ***********************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//// STL headers
#include <map>
#include <set>
#include <stack>
#include <deque>
#include <list>
#include <vector>

#include <string>

#include "AtomIdSet.hpp"
#include "Atom/AtomSet.hpp"
#include "CodePatterns/Cacheable.hpp"
#include "CodePatterns/Observer/Observable.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Fragmentation/HydrogenSaturation_enum.hpp"
#include "Formula.hpp"
#include "Helpers/defs.hpp"
#include "IdPool_policy.hpp"
#include "IdPool.hpp"
#include "Shapes/Shape.hpp"
#include "types.hpp"

/****************************************** forward declarations *****************************/

class atom;
class bond;
class BondedParticle;
class BondGraph;
class DepthFirstSearchAnalysis;
class element;
class ForceMatrix;
class Graph;
class LinkedCell_deprecated;
class ListOfLocalAtoms_t;
class molecule;
class MoleculeLeafClass;
class MoleculeListClass;
class MoleculeUnittest;
class RealSpaceMatrix;
class Vector;

/************************************* Class definitions ****************************************/

/** The complete molecule.
 * Class incorporates number of types
 */
class molecule : public Observable
{
  //!> grant unit test access
  friend class MoleculeUnittest;
  //!> function may access cstor
  friend molecule *NewMolecule();
  //!> function may access dstor
  friend void DeleteMolecule(molecule *);

public:
  typedef AtomIdSet::atomIdSet atomIdSet;
  typedef AtomIdSet::iterator iterator;
  typedef AtomIdSet::const_iterator const_iterator;

  int MDSteps; //!< The number of MD steps in Trajectories
  mutable int NoNonBonds; //!< number of non-hydrogen bonds in molecule
  mutable int NoCyclicBonds; //!< number of cyclic bonds in molecule, by DepthFirstSearchAnalysis()
  bool ActiveFlag; //!< in a MoleculeListClass used to discern active from inactive molecules
  int IndexNr; //!< index of molecule in a MoleculeListClass
  char name[MAXSTRINGSIZE]; //!< arbitrary name

private:
  Formula formula;
  Cacheable<size_t> NoNonHydrogen; //!< number of non-hydrogen atoms in molecule
  Cacheable<int> BondCount; //!< number of atoms, brought up-to-date by doCountBonds()
  moleculeId_t id;
  AtomIdSet atomIds; //<!set of atomic ids to check uniqueness of atoms
  IdPool<atomId_t, uniqueId> atomIdPool;  //!< pool of internal ids such that way may guarantee uniqueness

protected:

  molecule();
  virtual ~molecule();

public:

  /******* Notifications *******/

  //!> enumeration of present notification types: only insertion/removal of atoms or molecules
  enum NotificationType {
    AtomInserted,
    AtomRemoved,
    AtomNrChanged,
    MoleculeNameChanged,
    NotificationType_MAX
  };

public:
  //getter and setter
  const std::string getName() const;
  int getAtomCount() const;
  size_t doCountNoNonHydrogen() const;
  size_t getNoNonHydrogen() const;
  int getBondCount() const;
  int doCountBonds() const;
  moleculeId_t getId() const;
  void setId(moleculeId_t);
  void setName(const std::string);
  const Formula &getFormula() const;
  unsigned int getElementCount() const;
  bool hasElement(const element*) const;
  bool hasElement(atomicNumber_t) const;
  bool hasElement(const std::string&) const;

  virtual bool changeId(atomId_t newId);

  World::AtomComposite getAtomSet() const;

  // simply pass on all functions to AtomIdSet
  iterator begin() {
    return atomIds.begin();
  }
  const_iterator begin() const
  {
    return atomIds.begin();
  }
  iterator end()
  {
    return atomIds.end();
  }
  const_iterator end() const
  {
    return atomIds.end();
  }
  bool empty() const
  {
    return atomIds.empty();
  }
  size_t size() const
  {
    return atomIds.size();
  }
  const_iterator find(atom * key) const
  {
    return atomIds.find(key);
  }

  /** Returns the set of atomic ids contained in this molecule.
   *
   * @return set of atomic ids
   */
  const atomIdSet & getAtomIds() const {
    return atomIds.getAtomIds();
  }

  std::pair<iterator, bool> insert(atom * const key);

  /** Predicate whether given \a key is contained in this molecule.
   *
   * @param key atom to check
   * @return true - is contained, false - else
   */
  bool containsAtom(const atom* key) const
  {
    return atomIds.contains(key);
  }

  /** Predicate whether given \a id is contained in this molecule.
   *
   * @param id atomic id to check
   * @return true - is contained, false - else
   */
  bool containsAtom(const atomId_t id) const
  {
    return atomIds.contains(id);
  }

private:
  friend void atom::removeFromMolecule();
  /** Erase an atom from the list.
   * \note This should only be called by atom::removeFromMolecule(),
   * otherwise it is not assured that the atom knows about it.
   *
   * @param loc locator to atom in list
   * @return iterator to just after removed item (compliant with standard)
   */
  const_iterator erase(const_iterator loc);

  /** Erase an atom from the list.
   * \note This should only be called by atom::removeFromMolecule(),
   * otherwise it is not assured that the atom knows about it.
   *
   * @param *key key to atom in list
   * @return iterator to just after removed item (compliant with standard)
   */
  const_iterator erase(atom * key);

private:
  friend bool atom::changeNr(int newId);
  /**
   * used when changing an ParticleInfo::Nr.
   * Note that this number is local with this molecule.
   * Unless you are calling this method from inside an atom don't fiddle with the third parameter.
   *
   * @param oldNr old Nr
   * @param newNr new Nr to set
   * @param *target ref to atom
   * @return indicates wether the change could be done or not.
   */
  bool changeAtomNr(int oldNr, int newNr, atom* target=0);

  /** Sets the name of the atom.
   *
   * The name is set via its element symbol and its internal ParticleInfo::Nr.
   *
   * @param _atom atom whose name to set
   */
  void setAtomName(atom *_atom) const;

public:

  /** Function to create a bounding spherical shape for the currently associated atoms.
   *
   * \param boundary extra boundary of shape around (i.e. distance between outermost atom
   *        and the shape's surface)
   */
  Shape getBoundingShape(const double boundary = 0.) const;

  /// remove atoms from molecule.
  bool AddAtom(atom *pointer);
  bool RemoveAtom(atom *pointer);
  bool UnlinkAtom(atom *pointer);
  bool CleanupMolecule();
  void removeAtomsinMolecule();

  /// Add/remove atoms to/from molecule.
  atom * AddCopyAtom(atom *pointer);
  bool AddHydrogenReplacementAtom(bond::ptr Bond, atom *BottomOrigin, atom *TopOrigin, atom *TopReplacement, bool IsAngstroem);
  bond::ptr AddBond(atom *first, atom *second, int degree = 1);
  bool hasBondStructure() const;

  /// Find atoms.
  atom * FindAtom(int Nr) const;
  atom * AskAtom(std::string text);
  bool isInMolecule(const atom * const _atom);

  /// Count and change present atoms' coordination.
  bool CenterInBox();
  bool BoundInBox();
  void CenterEdge(Vector *max);
  void CenterOrigin();
  void CenterPeriodic();
  void CenterAtVector(Vector *newcenter);
  void Translate(const Vector *x);
  void TranslatePeriodically(const Vector *trans);
  void Mirror(const Vector *x);
  void Align(Vector *n);
  void Scale(const double ** const factor);
  void DeterminePeriodicCenter(Vector &center, const enum HydrogenTreatment _treatment = ExcludeHydrogen);
  Vector * DetermineCenterOfGravity() const;
  Vector * DetermineCenterOfAll() const;
  Vector * DetermineCenterOfBox() const;
  void SetNameFromFilename(const char *filename);
  void SetBoxDimension(Vector *dim);
  bool ScanForPeriodicCorrection();
  double VolumeOfConvexEnvelope(bool IsAngstroem);
  RealSpaceMatrix getInertiaTensor() const;
  void RotateToPrincipalAxisSystem(const Vector &Axis);

  bool CheckBounds(const Vector *x) const;
  void GetAlignvector(struct lsq_params * par) const;

  /// Initialising routines in fragmentation
  void OutputBondsList() const;

  bond::ptr CopyBond(atom *left, atom *right, bond::ptr CopyBond);

  molecule *CopyMolecule(const Vector &offset = zeroVec) const;
  molecule* CopyMoleculeFromSubRegion(const Shape&) const;

  /// Fragment molecule by two different approaches:
  bool StoreBondsToFile(std::string filename, std::string path = "");
  bool CreateFatherLookupTable(ListOfLocalAtoms_t &LookupTable, int count = 0);

  // Recognize doubly appearing molecules in a list of them
  int * GetFatherSonAtomicMap(molecule *OtherMolecule);
  bool FillBondStructureFromReference(const molecule * const reference, ListOfLocalAtoms_t &ListOfLocalAtoms, bool FreeList = false);
  bool FillListOfLocalAtoms(ListOfLocalAtoms_t &ListOfLocalAtoms, const int GlobalAtomCount);

  // Output routines.
  bool Output(std::ostream * const output) const;
  void OutputListOfBonds() const;

  // Manipulation routines
  void flipActiveFlag();

private:
  int last_atom; //!< number given to last atom
};

molecule *NewMolecule();
void DeleteMolecule(molecule* mol);



#endif /*MOLECULES_HPP_*/

