/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtMainWindow.cpp
 *
 *  Created on: Jan 14, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtMainWindow.hpp"

#include<Qt/qapplication.h>
#include<Qt/qlabel.h>
#include<Qt/qstring.h>
#include<Qt/qmenubar.h>
#include<Qt/qsplitter.h>
#include<Qt/qtoolbar.h>
#include<Qt/qsettings.h>
#include<Qt/qmessagebox.h>
#include<Qt/qboxlayout.h>
#include<Qt/qevent.h>

#include<iostream>
#include<map>

#include<boost/bind.hpp>

#include "Menu/Qt4/QtMenu.hpp"
#include "Views/Qt4/MoleculeList/QtMoleculeList.hpp"
#include "Views/Qt4/MoleculeList/QtMoleculeListView.hpp"
#include "Views/Qt4/ElementList/QtElementList.hpp"
#include "Views/Qt4/QtFragmentList.hpp"
#include "Views/Qt4/QtHomologyList.hpp"
#include "Views/Qt4/QtPotentialList.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedInstanceBoard.hpp"
#include "Views/Qt4/QtGeometryList.hpp"
#include "Views/Qt4/QtLogBox.hpp"
#include "Views/Qt4/QtShapeController.hpp"
#include "Views/Qt4/QtInfoBox.hpp"
#include "Views/Qt4/QtStatusBar.hpp"
#include "Views/Qt4/QtTimeLine.hpp"
#include "Views/Qt4/QtToolBar.hpp"
#include "Views/Qt4/Qt3D/GLWorldView.hpp"

//#include "CodePatterns/MemDebug.hpp"
#include "CodePatterns/Verbose.hpp"

#include "Atom/atom.hpp"
#include "molecule.hpp"
#include "Actions/Action.hpp"
#include "Actions/ActionQueue.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Actions/WorldAction/OutputAction.hpp"
#include "Actions/WorldAction/OutputAsAction.hpp"

#include "version.h"

using namespace MoleCuilder;

QtMainWindow::QtMainWindow(QApplication *_theApp) :
    theApp(_theApp)
{
  qRegisterMetaType<atomId_t>("atomId_t");
  qRegisterMetaType<moleculeId_t>("moleculeId_t");

  Q_INIT_RESOURCE(icons);
  setWindowTitle(MOLECUILDERVERSION);
  theApp->setWindowIcon(QIcon(QPixmap(":/molecuildergui_logo.png")));
  QCoreApplication::setOrganizationName("ins");
  QCoreApplication::setOrganizationDomain("ins.uni-bonn.de");
  QCoreApplication::setApplicationName("MoleCuilder");

  InstanceBoard = new QtObservedInstanceBoard(this);

  QSplitter *splitter1 = new QSplitter (Qt::Horizontal, this );
  QSplitter *splitter2 = new QSplitter (Qt::Vertical, splitter1 );
  QSplitter *splitter3 = new QSplitter (Qt::Vertical, splitter1 );
  QTabWidget *worldTab = new QTabWidget(splitter2);
  QWidget *layoutwidget = new QWidget(splitter2);
  QVBoxLayout *layout = new QVBoxLayout(layoutwidget);

  QtMoleculeListView *moleculeListView = new QtMoleculeListView(InstanceBoard);
  moleculeList = new QtMoleculeList(InstanceBoard, moleculeListView);
  moleculeListView->setModel(moleculeList);

  elementList = new QtElementList(InstanceBoard, worldTab);
  fragmentList = new QtFragmentList(worldTab);
  homologyList = new QtHomologyList(worldTab);
  potentialList = new QtPotentialList(worldTab);
  geometryList = new QtGeometryList(worldTab);
  logBox = new QtLogBox(std::cout, worldTab);
  errorlogBox = new QtLogBox(std::cerr, worldTab);
  shapeController = new QtShapeController(worldTab);

  timeline = new QtTimeLine(this);

  infoBox = new QtInfoBox(InstanceBoard);
  glWorldView = new GLWorldView(InstanceBoard, InstanceBoard);
  glWorldView->setSizePolicy( QSizePolicy::Expanding,QSizePolicy::Expanding);
  glWorldView->setMinimumSize( QSize(640,480) );
//  glWorldView->setFocus();
  glWorldView->camera()->setEye( QVector3D(0,3,10));

  MainMenu = new QtMenu<QMenuBar>(menuBar(), "");
  MainMenu->init();

  toolbar = new QtToolBar(this);
  glWorldView->addToolBarActions(toolbar);
  toolbar->addFavoriteActionItems(20);
  addToolBar(toolbar);

  // add context menu on right click
  glWorldView->setContextMenuPolicy(Qt::CustomContextMenu);
  connect(glWorldView, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(slotContextMenuRequested(const QPoint&)));

  setCentralWidget(splitter1);
  splitter1->addWidget(splitter2);
  splitter1->addWidget(splitter3);
  splitter1->setStretchFactor(0, 10);
  splitter3->addWidget(infoBox);
  splitter3->addWidget(shapeController);
  splitter2->addWidget(glWorldView);
  layout->addWidget(timeline);
  layout->addWidget(worldTab);
  splitter2->addWidget(layoutwidget);
  worldTab->addTab(moleculeListView, "Molecules");
  worldTab->addTab(elementList, "All Elements");
  worldTab->addTab(fragmentList, "All Fragments");
  worldTab->addTab(homologyList, "All Homologies");
  worldTab->addTab(potentialList, "All Potentials");
  worldTab->addTab(geometryList, "All Geometries");
  worldTab->addTab(logBox, "Log");
  worldTab->addTab(errorlogBox, "Errors");

  statusBar = new QtStatusBar(this);
  setStatusBar(statusBar);

  QSettings settings;
  settings.beginGroup("MainWindow");
  settingsSize = settings.value("size", QSize(400, 400)).toSize();
  settingsPosition = settings.value("position", QPoint(200, 200)).toPoint();

  const QStringList args = QCoreApplication::arguments();
  LOG(2, "DEBUG: QApp's seen arguments are " << args.join(",").toStdString());

  const int indexSize = args.indexOf(QString("--qt_size"));
  const int indexPosition = args.indexOf(QString("--qt_position"));

  if ((indexSize == -1) || (indexPosition == -1)) {
    resize(settingsSize);
    move(settingsPosition);
  } else {
    // Take size and position from command-line values instead
    // NOTE: Qt 4.8's toInt() caps the second argument when it's below 1000 (yes, really!). Hence, we're using stoi()
    const QSize cmdlineSize = QSize(stoi(args[indexSize + 1].split(",")[0].toStdString()), stoi(args[indexSize + 1].split(",")[1].toStdString()));
    const QPoint cmdlinePosition = QPoint(stoi(args[indexPosition + 1].split(",")[0].toStdString()), stoi(args[indexPosition + 1].split(",")[1].toStdString()));
    LOG(2, "DEBUG: Using size and position from the command line: (" << cmdlineSize.width() << "," << cmdlineSize.height()
        << "), (" << cmdlinePosition.x() << "," << cmdlinePosition.y() << ")");
    resize(cmdlineSize);
    move(cmdlinePosition);
  }
  if (settings.value("maximized", false).toBool())
    showMaximized();
  settings.endGroup();
  LOG(1, "INFO: Size and position is (" << size().width() << "," << size().height() << "), (" << pos().x() << "," << pos().y() << ")");

  connect(glWorldView,SIGNAL(hoverChanged(const atomId_t)), infoBox,SLOT(atomHover(const atomId_t)));
  connect(glWorldView,SIGNAL(hoverChanged(const moleculeId_t, int)), infoBox,SLOT(moleculeHover(const moleculeId_t)));
  connect(moleculeList,SIGNAL(moleculesVisibilityChanged(ObservedValue_Index_t,bool)),
      glWorldView, SIGNAL(moleculesVisibilityChanged(ObservedValue_Index_t,bool)));

  glWorldView->fitCameraToDomain();
}

void QtMainWindow::slotContextMenuRequested(const QPoint &pos)
{
  // pick the atoms menu
  QtMenu<QMenu_tooltip> * atoms_submenu = MainMenu->findSubmenu("atom");
  // and show it
  if (atoms_submenu != NULL) {
    atoms_submenu->popup(glWorldView->mapToGlobal(pos));
  } else {
    ELOG(1, "Could not find atom submenu.");
  }
}
QtMainWindow::~QtMainWindow()
{
  const QStringList args = QCoreApplication::arguments();
  const bool containsSize = args.contains(QString("--qt_size"));
  const bool containsPosition = args.contains(QString("--qt_position"));

  QSettings settings;
  settings.beginGroup("MainWindow");
  if (!containsSize || !containsPosition) {
    settings.setValue("size", size());
    settings.setValue("position", pos());
  } else {
    settings.setValue("size", settingsSize);
    settings.setValue("position", settingsPosition);
  }
  settings.setValue("maximized", isMaximized());
  settings.endGroup();

  menuBar()->clear();
  delete MainMenu;
}

void QtMainWindow::display() {
  this->show();
  this->update();
  theApp->exec();
}

void QtMainWindow::closeEvent(QCloseEvent *event)
{
  if (ChangeTracker::getInstance().hasChanged()){
    int ret = QMessageBox::question(this, tr("MoleCuilder"),
                                    tr("The world has been modified.\n"
                                       "Do you want to save your changes?"),
                                    QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel,
                                    QMessageBox::Save);
    switch(ret){
      case QMessageBox::Cancel:
        event->ignore();
        return;
      case QMessageBox::Save:
        if (FormatParserStorage::getInstance().isAbleToSave())
          MoleCuilder::WorldOutput();
        else{
          try{
            ActionQueue::getInstance().queueAction("output-as");
          }catch(...){
            ELOG(1, "Action failed or cancelled");
          }
        }
        break;
    }
  }
  QMainWindow::closeEvent(event);
}

