/*
 * PrincipalAxisSystemAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include "Actions/AnalysisAction/PrincipalAxisSystemAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "element.hpp"
#include "molecule.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

const char AnalysisPrincipalAxisSystemAction::NAME[] = "principal-axis-system";

AnalysisPrincipalAxisSystemAction::AnalysisPrincipalAxisSystemAction() :
  Action(NAME)
{}

AnalysisPrincipalAxisSystemAction::~AnalysisPrincipalAxisSystemAction()
{}

void AnalysisPrincipalAxisSystem() {
  ActionRegistry::getInstance().getActionByName(AnalysisPrincipalAxisSystemAction::NAME)->call(Action::NonInteractive);
};

Dialog* AnalysisPrincipalAxisSystemAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryEmpty(NAME, ValueStorage::getInstance().getDescription(NAME));

  return dialog;
}

Action::state_ptr AnalysisPrincipalAxisSystemAction::performCall() {
  molecule *mol = NULL;
  Matrix InertiaTensor;

  ValueStorage::getInstance().queryCurrentValue(NAME, mol);
  DoLog(0) && (Log() << Verbose(0) << "Evaluating prinicipal axis." << endl);
  for (World::MoleculeSelectionIterator iter = World::getInstance().beginMoleculeSelection(); iter != World::getInstance().endMoleculeSelection(); ++iter) {
    molecule *mol = iter->second;
    Vector *CenterOfGravity = mol->DetermineCenterOfGravity();

    // reset inertia tensor
    InertiaTensor.zero();

    // sum up inertia tensor
    for (molecule::const_iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      Vector x = (*iter)->getPosition();
      x -= *CenterOfGravity;
      const double mass = (*iter)->getType()->mass;
      InertiaTensor.at(0,0) += mass*(x[1]*x[1] + x[2]*x[2]);
      InertiaTensor.at(0,1) += mass*(-x[0]*x[1]);
      InertiaTensor.at(0,2) += mass*(-x[0]*x[2]);
      InertiaTensor.at(1,0) += mass*(-x[1]*x[0]);
      InertiaTensor.at(1,1) += mass*(x[0]*x[0] + x[2]*x[2]);
      InertiaTensor.at(1,2) += mass*(-x[1]*x[2]);
      InertiaTensor.at(2,0) += mass*(-x[2]*x[0]);
      InertiaTensor.at(2,1) += mass*(-x[2]*x[1]);
      InertiaTensor.at(2,2) += mass*(x[0]*x[0] + x[1]*x[1]);
    }
    // print InertiaTensor for debugging
    DoLog(0) && (Log() << Verbose(0) << "The inertia tensor is:" << InertiaTensor << endl);
  }
  return Action::success;
}

Action::state_ptr AnalysisPrincipalAxisSystemAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr AnalysisPrincipalAxisSystemAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool AnalysisPrincipalAxisSystemAction::canUndo() {
  return true;
}

bool AnalysisPrincipalAxisSystemAction::shouldUndo() {
  return true;
}

const string AnalysisPrincipalAxisSystemAction::getName() {
  return NAME;
}
