#ifndef MOLECULEDESCRIPTOR_IMPL_HPP
#define MOLECULEDESCRIPTOR_IMPL_HPP

#include "Descriptors/MoleculeDescriptor.hpp"

/************************ Declarations of implementation Objects ************************/

/**
 * This class implements a general Base class for MoleculeDescriptors using the PIMPL-Idiom
 *
 * The predicate for this class is empty and should be implemented by derived classes.
 * By the predicate it is described which molecules should be picked for a given descriptor.
 */

class MoleculeDescriptor_impl
{
  friend class MoleculeDescriptor;
public:

  MoleculeDescriptor_impl();
  virtual ~MoleculeDescriptor_impl();

  /**
   * Implement this abstract Method to make a concrete MoleculeDescriptor pick certain Molecules
   */
  virtual bool predicate(std::pair<moleculeId_t,molecule*>)=0;

protected:

  /**
   * This method is called when the Descriptor is used to find the first matching
   * Molecule. Walks through all Molecules and stops on the first match. Can be implemented
   * when the searched Molecule can be found in a more efficient way. Calculated
   * Moleculedescriptors will always use this method, so no improvement there.
   */
  virtual molecule* find();

  /**
   * This method is called when the Descriptor is used to find all matching Molecules.
   * Walks through all Molecules and tests the predicate on each one. A vector of all
   * matching Molecules is returned.
   */
  virtual std::vector<molecule*> findAll();

  /**
   * This method is used internally to query the Set of Molecules from the world.
   * By using this method derived classes can also access the Internal World Datastructre.
   * Implemented in full in the Base Descriptor Implementation, so only this one method
   * needs to be friend with the World class.
   */
  World::MoleculeSet& getMolecules();

  void checkAndAdd(std::vector<molecule*>*,std::pair<moleculeId_t,molecule*>);
};

/************************** Universe and Emptyset *****************/

/**
 * A simple MoleculeDescriptor that will always match all Molecules present in the World.
 */
class MoleculeAllDescriptor_impl : public MoleculeDescriptor_impl {
public:
  MoleculeAllDescriptor_impl();
  virtual ~MoleculeAllDescriptor_impl();

  /**
   * Always returns true for any Molecule
   */
  virtual bool predicate(std::pair<moleculeId_t,molecule*>);
};


/**
 * An MoleculeDescriptor that never matches any Molecule in the World.
 */
class MoleculeNoneDescriptor_impl : public MoleculeDescriptor_impl {
public:
  MoleculeNoneDescriptor_impl();
  virtual ~MoleculeNoneDescriptor_impl();

  /**
   * Always returns false for any Molecule
   */
  virtual bool predicate(std::pair<moleculeId_t,molecule*>);
};

/************************** Operator stuff ************************/

/**
 * Intersection of two MoleculeDescriptors
 */
class MoleculeAndDescriptor_impl : public MoleculeDescriptor_impl
{
public:
  MoleculeAndDescriptor_impl(MoleculeDescriptor::impl_ptr _lhs, MoleculeDescriptor::impl_ptr _rhs);
  ~MoleculeAndDescriptor_impl();

  /**
   * This predicate uses the predicate from the first && the predicate from the
   * second Descriptor to decide if an Molecule should be selected.
   */
  virtual bool predicate(std::pair<moleculeId_t,molecule*>);

private:
  MoleculeDescriptor::impl_ptr lhs;
  MoleculeDescriptor::impl_ptr rhs;
};

/**
 * Union of two MoleculeDescriptors
 */
class MoleculeOrDescriptor_impl : public MoleculeDescriptor_impl
{
public:
  MoleculeOrDescriptor_impl(MoleculeDescriptor::impl_ptr _lhs, MoleculeDescriptor::impl_ptr _rhs);
  virtual ~MoleculeOrDescriptor_impl();

  /**
   * This predicate uses the predicate form the first || the predicate from the
   * second Descriptor to decide if an Molecule should be selected.
   */
  virtual bool predicate(std::pair<moleculeId_t,molecule*>);

private:
  MoleculeDescriptor::impl_ptr lhs;
  MoleculeDescriptor::impl_ptr rhs;
};

/**
 * Set Inversion of a Descriptor
 */
class MoleculeNotDescriptor_impl : public MoleculeDescriptor_impl
{
public:
  MoleculeNotDescriptor_impl(MoleculeDescriptor::impl_ptr _arg);
  virtual ~MoleculeNotDescriptor_impl();

  /**
   * Opposite of the given descriptor predicate.
   */
  virtual bool predicate(std::pair<moleculeId_t,molecule*>);

private:
  MoleculeDescriptor::impl_ptr arg;
};

#endif //MOLECULEDESCRIPTOR_IMPL_HPP
