/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MenuDescription.cpp
 *
 *  Created on: Oct 26, 2010
 *      Author: heber
 */

#include <map>
#include <string>

#include "Menu/MenuDescription.hpp"

/** Constructor of class MenuDescription.
 *
 */
MenuDescription::MenuDescription()
{
  // put each menu into set
  MenuSet.insert("analysis");
  MenuSet.insert("atom");
  MenuSet.insert("command");
  MenuSet.insert("fragmentation");
  MenuSet.insert("molecule");
  MenuSet.insert("parser");
  MenuSet.insert("selection");
  MenuSet.insert("tesselation");
  MenuSet.insert("world");

  // put menu description into each menu category
  MenuDescriptionsMap["analysis"] = "Analysis (pair correlation, volume)";
  MenuDescriptionsMap["atom"] = "Edit atoms";
  MenuDescriptionsMap["command"] = "Configuration";
  MenuDescriptionsMap["fragmentation"] = "Fragmentation";
  MenuDescriptionsMap["molecule"] = "Parse files into system";
  MenuDescriptionsMap["parser"] = "Edit molecules (load, parse, save)";
  MenuDescriptionsMap["selection"] = "Select atoms/molecules";
  MenuDescriptionsMap["tesselation"] = "Tesselate molecules";
  MenuDescriptionsMap["world"] = "Edit world";

  // put menu name into each menu category
  MenuNameMap["analysis"] = "Analysis";
  MenuNameMap["atom"] = "Atoms";
  MenuNameMap["command"] = "configuration options";
  MenuNameMap["fragmentation"] = "Fragmentation";
  MenuNameMap["molecule"] = "Molecules";
  MenuNameMap["parser"] = "Input/Output";
  MenuNameMap["selection"] = "Selection";
  MenuNameMap["tesselation"] = "Tesselation";
  MenuNameMap["world"] = "Globals";
}

/** Destructor of class MenuDescription.
 *
 */
MenuDescription::~MenuDescription()
{}

/** Getter for MenuDescriptionsMap.
 * \param token name of menu
 * \return description string of the menu or empty
 */
const std::string MenuDescription::getDescription(std::string token) const
{
  if (MenuDescriptionsMap.find(token) != MenuDescriptionsMap.end())
    return MenuDescriptionsMap.find(token)->second;
  else
    return std::string();
}

/** Getter for MenuNameMap.
 * \param token name of menu
 * \return description string of the menu or empty
 */
const std::string MenuDescription::getName(std::string token) const
{
  if (MenuNameMap.find(token) != MenuNameMap.end())
    return MenuNameMap.find(token)->second;
  else
    return std::string();
}

/** Forward iterator from beginning of list of descriptions.
 * \return iterator
 */
MenuDescription::iterator MenuDescription::getBeginIter()
{
  return MenuSet.begin();
}

/** Forward iterator at end of list of descriptions.
 * \return iterator
 */
MenuDescription::iterator MenuDescription::getEndIter()
{
  return MenuSet.end();
}

/** Constant forward iterator from beginning of list of descriptions.
 * \return constant iterator
 */
MenuDescription::const_iterator MenuDescription::getBeginIter() const
{
  return MenuSet.begin();
}

/** Constant forward iterator at end of list of descriptions.
 * \return constant iterator
 */
MenuDescription::const_iterator MenuDescription::getEndIter() const
{
  return MenuSet.end();
}


