/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GLWorldScene.cpp
 *
 *  This is based on the Qt3D example "teaservice", specifically parts of teaservice.cpp.
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLWorldScene.hpp"
#include <Qt3D/qglview.h>
#include <Qt3D/qglbuilder.h>
#include <Qt3D/qglscenenode.h>
#include <Qt3D/qglsphere.h>
#include <Qt3D/qglcylinder.h>

#include "GLMoleculeObject.hpp"
#include "GLMoleculeObject_atom.hpp"
#include "GLMoleculeObject_bond.hpp"
#include "GLMoleculeObject_molecule.hpp"
#include "GLMoleculeObject_shape.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"

#include "Actions/SelectionAction/Atoms/AtomByIdAction.hpp"
#include "Actions/SelectionAction/Atoms/NotAtomByIdAction.hpp"
#include "Actions/SelectionAction/Molecules/MoleculeByIdAction.hpp"
#include "Actions/SelectionAction/Molecules/NotMoleculeByIdAction.hpp"
#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Helpers/helpers.hpp"
#include "Shapes/ShapeRegistry.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>

using namespace MoleCuilder;

GLWorldScene::GLWorldScene(QObject *parent)
   : QObject(parent)
{
  int sphereDetails[] = {5, 3, 2, 0};
  int cylinderDetails[] = {16, 8, 6, 3};
  for (int i=0;i<GLMoleculeObject::DETAILTYPES_MAX;i++){
    QGLBuilder emptyBuilder;
    GLMoleculeObject::meshEmpty[i] = emptyBuilder.finalizedSceneNode();
    QGLBuilder sphereBuilder;
    sphereBuilder << QGLSphere(2.0, sphereDetails[i]);
    GLMoleculeObject::meshSphere[i] = sphereBuilder.finalizedSceneNode();
    GLMoleculeObject::meshSphere[i]->setOption(QGLSceneNode::CullBoundingBox, true);
    QGLBuilder cylinderBuilder;
    cylinderBuilder << QGLCylinder(.25,.25,1.0,cylinderDetails[i]);
    GLMoleculeObject::meshCylinder[i] = cylinderBuilder.finalizedSceneNode();
    GLMoleculeObject::meshCylinder[i]->setOption(QGLSceneNode::CullBoundingBox, true);
  }

  connect(this, SIGNAL(updated()), this, SLOT(update()));


  setSelectionMode(SelectAtom);

  init();
}

GLWorldScene::~GLWorldScene()
{
  // remove all elements
  GLMoleculeObject::cleanMaterialMap();
}

/** Initialise the WorldScene with molecules and atoms from World.
 *
 */
void GLWorldScene::init()
{
//  const std::vector<const molecule *> &molecules =
//      const_cast<const World &>(World::getInstance()).getAllMolecules();
//
//  for (std::vector<const molecule*>::const_iterator moliter = molecules.begin();
//      moliter != molecules.end();
//      moliter++) {
//    // create molecule objects in scene
//    moleculeInserted((*moliter)->getId());
//  }
}

/** Update the WorldScene with molecules and atoms from World.
 *
 * This function should be called after e.g. WorldTime::TimeChanged was
 * received or after another molecule has been loaded.
 *
 */
void GLWorldScene::update()
{
  const std::vector<const molecule *> &molecules =
      const_cast<const World &>(World::getInstance()).getAllMolecules();

  for (std::vector<const molecule*>::const_iterator moliter = molecules.begin();
      moliter != molecules.end();
      moliter++) {
    // check whether molecule already exists
    const moleculeId_t molid = (*moliter)->getId();
    const bool mol_present = MoleculesinSceneMap.count(molid);
    if (!mol_present)
      moleculeInserted((*moliter)->getId());
  }

  MoleculeNodeMap::iterator iter = MoleculesinSceneMap.begin();
  for (;iter != MoleculesinSceneMap.end();) {
    const moleculeId_t molid = iter->first;
    const molecule * const mol = const_cast<const World &>(World::getInstance()).
        getMolecule(MoleculeById(molid));
    const bool mol_absent = (mol == NULL);
    // step on to next molecule before possibly removing entry and invalidating iter
    ++iter;
    if (mol_absent)
      moleculeRemoved(molid);
  }

}

void GLWorldScene::atomClicked(atomId_t no)
{
   LOG(3, "INFO: GLMoleculeObject_molecule - atom " << no << " has been clicked.");
   const atom * const Walker = const_cast<const World &>(World::getInstance()).
       getAtom(AtomById(no));
   if (selectionMode == SelectAtom){
     if (!World::getInstance().isSelected(Walker))
       SelectionAtomById(std::vector<atomId_t>(1,no));
     else
       SelectionNotAtomById(std::vector<atomId_t>(1,no));
   }else if (selectionMode == SelectMolecule){
     const molecule *mol = Walker->getMolecule();
     ASSERT(mol, "Atom without molecule has been clicked.");
     molids_t ids(1, mol->getId());
     if (!World::getInstance().isSelected(mol))
       SelectionMoleculeById(ids);
     else
       SelectionNotMoleculeById(ids);
   }
   emit clicked(no);
}

void GLWorldScene::moleculeClicked(moleculeId_t no)
{
   LOG(3, "INFO: GLMoleculeObject_molecule - mol " << no << " has been clicked.");
   const molecule * const mol= const_cast<const World &>(World::getInstance()).
       getMolecule(MoleculeById(no));
   ASSERT(mol, "Atom without molecule has been clicked.");
   molids_t ids(1, mol->getId());
   if (!World::getInstance().isSelected(mol))
     SelectionMoleculeById(ids);
   else
     SelectionNotMoleculeById(ids);
   emit clicked(no);
}

/** ....
 *
 */
void GLWorldScene::worldSelectionChanged()
{
  LOG(3, "INFO: GLWorldScene: Received signal selectionChanged.");

  const std::vector<const molecule*> &molecules =
      const_cast<const World &>(World::getInstance()).getAllMolecules();

  if (molecules.size() > 0) {
    for (std::vector<const molecule*>::const_iterator Runner = molecules.begin();
        Runner != molecules.end();
        Runner++) {

      // molecule selected but not in scene?
      const bool isSelected =
          const_cast<const World &>(World::getInstance()).isSelected(*Runner);
      if (isSelected){
        MoleculeNodeMap::iterator iter = MoleculesinSceneMap.find((*Runner)->getId());
        ASSERT( iter != MoleculesinSceneMap.end(),
            "GLWorldScene::worldSelectionChanged() - selected molecule is unknown.");
        GLMoleculeObject_molecule *molObject = iter->second;
        // inform molecule object
        molObject->selected(isSelected);
      }
    }
  }
}

/** Inserts a molecule into the scene.
 *
 * @param _mol molecule to insert
 */
void GLWorldScene::moleculeInserted(const moleculeId_t _id)
{
  LOG(3, "INFO: GLWorldScene: Received signal moleculeInserted for molecule "+toString(_id)+".");
  MoleculeNodeMap::const_iterator iter = MoleculesinSceneMap.find(_id);
  ASSERT( iter == MoleculesinSceneMap.end(),
      "GLWorldScene::moleculeInserted() - molecule's id "+toString(_id)+" already present.");

  // add new object
  GLMoleculeObject_molecule *molObject = new GLMoleculeObject_molecule(GLMoleculeObject::meshEmpty, this, _id);
  ASSERT( molObject != NULL,
      "GLWorldScene::moleculeInserted - could not create molecule object for "+toString(_id));
  MoleculesinSceneMap.insert( make_pair(_id, molObject) );
  connect (molObject, SIGNAL(changed()), this, SIGNAL(changed()));
  connect (molObject, SIGNAL(changeOccured()), this, SIGNAL(changeOccured()));
  connect (molObject, SIGNAL(atomClicked(atomId_t)), this, SLOT(atomClicked(atomId_t)));
  connect (molObject, SIGNAL(moleculeClicked(moleculeId_t)), this, SLOT(moleculeClicked(moleculeId_t)));
  connect (molObject, SIGNAL(selectionChanged()), this, SIGNAL(changed()));
  connect (molObject, SIGNAL(selectionChanged()), this, SIGNAL(changed()));
  connect (molObject, SIGNAL(hoverChanged(const atomId_t)), this, SIGNAL(hoverChanged(const atomId_t)));
  connect (molObject, SIGNAL(hoverChanged(const moleculeId_t, int)), this, SIGNAL(hoverChanged(const moleculeId_t, int)));
  emit changed();
  emit changeOccured();
}

/** Removes a molecule from the scene.
 *
 * @param _id id of molecule to remove
 */
void GLWorldScene::moleculeRemoved(const moleculeId_t _id)
{
  LOG(3, "INFO: GLWorldScene: Received signal moleculeRemoved for molecule "+toString(_id)+".");
  MoleculeNodeMap::iterator iter = MoleculesinSceneMap.find(_id);
  ASSERT( iter != MoleculesinSceneMap.end(),
      "GLWorldScene::moleculeInserted() - molecule's id "+toString(_id)+" is unknown.");

  GLMoleculeObject_molecule *molObject = iter->second;
  molObject->disconnect();
  MoleculesinSceneMap.erase(iter);
  delete molObject;
  emit changed();
  emit changeOccured();
}

void GLWorldScene::moleculesVisibilityChanged(const moleculeId_t _id, bool _visible)
{
  MoleculeNodeMap::iterator iter = MoleculesinSceneMap.find(_id);
  ASSERT( iter != MoleculesinSceneMap.end(),
      "GLWorldScene::moleculeInserted() - molecule's id "+toString(_id)+" is unknown.");

  GLMoleculeObject_molecule *molObject = iter->second;
  molObject->setVisible(_visible);

  emit changed();
  emit changeOccured();
}

/** Adds a shape to the scene.
 *
 * uses ShapeRegistry::lastChanged()
 *
 */
void GLWorldScene::addShape()
{
  Shape &shape = *ShapeRegistry::getInstance().lastChanged();
  GLMoleculeObject_shape *shapeObject = new GLMoleculeObject_shape(shape, this);
  ShapeNodeMap::iterator iter = ShapesinSceneMap.find(shape.getName());
  ASSERT(iter == ShapesinSceneMap.end(),
      "GLWorldScene::addShape() - same shape "+shape.getName()+" added again.");
  ShapesinSceneMap.insert( make_pair(shape.getName(), shapeObject) );
}

void GLWorldScene::removeShape()
{
  Shape &shape = *ShapeRegistry::getInstance().lastChanged();
  ShapeNodeMap::iterator iter = ShapesinSceneMap.find(shape.getName());
  ASSERT(iter != ShapesinSceneMap.end(),
      "GLWorldScene::removeShape() - shape "+shape.getName()+" not in scene.");
  ShapesinSceneMap.erase(iter);
  delete(iter->second);
}

void GLWorldScene::updateSelectedShapes()
{
  foreach (QObject *obj, children()) {
    GLMoleculeObject_shape *shapeobj = qobject_cast<GLMoleculeObject_shape *>(obj);
      if (shapeobj){
        shapeobj->enable(ShapeRegistry::getInstance().isSelected(shapeobj->getShape()));
      }
  }
}

void GLWorldScene::initialize(QGLView *view, QGLPainter *painter) const
{
  // Initialize all of the mesh objects that we have as children.
   foreach (QObject *obj, children()) {
     GLMoleculeObject *meshobj = qobject_cast<GLMoleculeObject *>(obj);
       if (meshobj)
         meshobj->initialize(view, painter);
   }
}

void GLWorldScene::draw(QGLPainter *painter, const QVector4D &cameraPlane) const
{
   // Draw all of the mesh objects that we have as children.
   foreach (QObject *obj, children()) {
     GLMoleculeObject *meshobj = qobject_cast<GLMoleculeObject *>(obj);
       if (meshobj)
         meshobj->draw(painter, cameraPlane);
   }
}

void GLWorldScene::setSelectionMode(SelectionModeType mode)
{
  selectionMode = mode;
  // TODO send update to toolbar
}

void GLWorldScene::setSelectionModeAtom()
{
  setSelectionMode(SelectAtom);
}

void GLWorldScene::setSelectionModeMolecule()
{
  setSelectionMode(SelectMolecule);
}
