/*
 * Dialog.hpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

#ifndef DIALOG_HPP_
#define DIALOG_HPP_

#include<string>
#include<list>
#include<vector>

class atom;
class Box;
class element;
class molecule;
class Vector;

class Dialog
{
public:
  Dialog();
  virtual ~Dialog();

  virtual void queryEmpty(const char *, std::string = "")=0;
  virtual void queryBoolean(const char *, bool *, std::string = "")=0;
  virtual void queryInt(const char *, int *, std::string = "")=0;
  virtual void queryDouble(const char*,double *, std::string = "")=0;
  virtual void queryString(const char*, std::string *, std::string = "")=0;
  virtual void queryAtom(const char*,atom**,std::string = "")=0;
  virtual void queryMolecule(const char*,molecule**, std::string = "")=0;
  virtual void queryVector(const char*,Vector *,bool, std::string = "")=0;
  virtual void queryBox(const char*,Box*, std::string = "")=0;
  virtual void queryElement(const char*, std::vector<element *> *, std::string = "")=0;

  virtual bool display();

protected:
  // methodology for handling queries
  // all queries are stored and then performed at appropriate times

  //these queries can be handled by this dialog

  //TODO: Find a way to reduce complexity...
  //needs O(N*M) query classes, where N is the number of query types and M is the number of GUIs
  //usual approach for reducing inheritance complexity (strategy pattern) does not work,
  //due to lack of common code for query types as well as GUI-Types (all subtypes differ a lot)

  //base class for all queries
  class Query {
    friend class Dialog;
  public:
    Query(std::string _title, std::string _description = "");
    virtual ~Query();
    virtual bool handle()=0;
    virtual void setResult()=0;
  protected:
    const std::string getTitle() const;
    const std::string getDescription() const;
  private:
    std::string title;  //!< short title of the query
    std::string description; //!< longer description for tooltips or for help
  };

  // Empty Query is just meant for showing text, such as version, help, initial message or alike
  class EmptyQuery : public Query {
  public:
    EmptyQuery(std::string title, std::string _description = "");
    virtual ~EmptyQuery();
    virtual bool handle()=0;
    virtual void setResult();
  };

  //Specialized classes for certain types. GUI-Types are not specialized at this time
  class BooleanQuery : public Query {
  public:
    BooleanQuery(std::string title,bool *_target, std::string _description = "");
    virtual ~BooleanQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    bool tmp;
  private:
    bool *target;
  };

  class IntQuery : public Query {
  public:
    IntQuery(std::string title,int *_target, std::string _description = "");
    virtual ~IntQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    int tmp;
  private:
    int *target;
  };

  class DoubleQuery : public Query {
  public:
    DoubleQuery(std::string title,double *_target, std::string _description = "");
    virtual ~DoubleQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    double tmp;
  private:
    double *target;
  };

  class StringQuery : public Query {
  public:
    StringQuery(std::string title,std::string *_target, std::string _description = "");
    virtual ~StringQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    std::string tmp;
  private:
    std::string *target;
  };

  class MoleculeQuery : public Query {
  public:
    MoleculeQuery(std::string title, molecule **_target, std::string _description = "");
    virtual ~MoleculeQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    molecule *tmp;
  private:
    molecule **target;
  };

  class AtomQuery : public Query {
  public:
    AtomQuery(std::string title, atom **_target, std::string _description = "");
    virtual ~AtomQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    atom *tmp;
  private:
    atom **target;
  };

  class VectorQuery : public Query {
  public:
      VectorQuery(std::string title,Vector *_target,bool _check, std::string _description = "");
      virtual ~VectorQuery();
      virtual bool handle()=0;
      virtual void setResult();
    protected:
      Vector *tmp;
      bool check;
    private:
      Vector *target;
  };

  class BoxQuery : public Query {
  public:
      BoxQuery(std::string title,Box *_cellSize, std::string _description = "");
      virtual ~BoxQuery();
      virtual bool handle()=0;
      virtual void setResult();
    protected:
      double* tmp;
    private:
      Box* target;
  };

  class ElementQuery : public Query {
  public:
    ElementQuery(std::string title, std::vector<element *> *_target, std::string _description = "");
    virtual ~ElementQuery();
    virtual bool handle()=0;
    virtual void setResult();
  protected:
    std::vector<element *> elements;
  private:
    std::vector<element *> * const target;
  };

void registerQuery(Query* query);

private:
  std::list<Query*> queries;

};


#endif /* DIALOG_HPP_ */
