/*
 * CommandLineDialog.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include <iostream>
#include <vector>

#include <Descriptors/AtomDescriptor.hpp>
#include <Descriptors/AtomIdDescriptor.hpp>
#include <Descriptors/MoleculeDescriptor.hpp>
#include <Descriptors/MoleculeIdDescriptor.hpp>
#include "CommandLineUI/CommandLineDialog.hpp"

#include "Actions/Values.hpp"

#include "element.hpp"
#include "periodentafel.hpp"
#include "CommandLineParser.hpp"
#include "defs.hpp"
#include "log.hpp"
#include "periodentafel.hpp"
#include "verbose.hpp"
#include "World.hpp"
#include "Box.hpp"

#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "vector.hpp"

using namespace std;


CommandLineDialog::CommandLineDialog()
{
}

CommandLineDialog::~CommandLineDialog()
{
}


void CommandLineDialog::queryEmpty(const char* title, string _description){
  registerQuery(new EmptyCommandLineQuery(title, _description));
}

void CommandLineDialog::queryInt(const char* title, int* target, string _description){
  registerQuery(new IntCommandLineQuery(title,target, _description));
}

void CommandLineDialog::queryBoolean(const char* title, bool* target, string _description){
  registerQuery(new BooleanCommandLineQuery(title,target, _description));
}

void CommandLineDialog::queryDouble(const char* title, double* target, string _description){
  registerQuery(new DoubleCommandLineQuery(title,target, _description));
}

void CommandLineDialog::queryString(const char* title, string* target, string _description){
  registerQuery(new StringCommandLineQuery(title,target, _description));
}

void CommandLineDialog::queryAtom(const char* title, atom **target, string _description) {
  registerQuery(new AtomCommandLineQuery(title,target, _description));
}

void CommandLineDialog::queryMolecule(const char* title, molecule **target, string _description) {
  registerQuery(new MoleculeCommandLineQuery(title,target, _description));
}

void CommandLineDialog::queryVector(const char* title, Vector *target, bool check, string _description) {
  registerQuery(new VectorCommandLineQuery(title,target,check, _description));
}

void CommandLineDialog::queryBox(const char* title, Box* cellSize, string _description) {
  registerQuery(new BoxCommandLineQuery(title,cellSize,_description));
}

void CommandLineDialog::queryElement(const char* title, std::vector<element *> *target, string _description){
  registerQuery(new ElementCommandLineQuery(title,target, _description));
}

/************************** Query Infrastructure ************************/

CommandLineDialog::EmptyCommandLineQuery::EmptyCommandLineQuery(string title, string _description) :
    Dialog::EmptyQuery(title, _description)
{}

CommandLineDialog::EmptyCommandLineQuery::~EmptyCommandLineQuery() {}

bool CommandLineDialog::EmptyCommandLineQuery::handle() {
  cout << "Message of " << getTitle() << ":\n" << getDescription() << "\n";
  return true;
}

CommandLineDialog::IntCommandLineQuery::IntCommandLineQuery(string title,int *_target, string _description) :
    Dialog::IntQuery(title,_target, _description)
{}

CommandLineDialog::IntCommandLineQuery::~IntCommandLineQuery() {}

bool CommandLineDialog::IntCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<int>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing integer for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::BooleanCommandLineQuery::BooleanCommandLineQuery(string title,bool *_target, string _description) :
    Dialog::BooleanQuery(title,_target, _description)
{}

CommandLineDialog::BooleanCommandLineQuery::~BooleanCommandLineQuery() {}

bool CommandLineDialog::BooleanCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<bool>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing boolean for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::StringCommandLineQuery::StringCommandLineQuery(string title,string *_target, string _description) :
    Dialog::StringQuery(title,_target, _description)
{}

CommandLineDialog::StringCommandLineQuery::~StringCommandLineQuery() {}

bool CommandLineDialog::StringCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<string>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing string for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::DoubleCommandLineQuery::DoubleCommandLineQuery(string title,double *_target, string _description) :
    Dialog::DoubleQuery(title,_target, _description)
{}

CommandLineDialog::DoubleCommandLineQuery::~DoubleCommandLineQuery() {}

bool CommandLineDialog::DoubleCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<double>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing double for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::AtomCommandLineQuery::AtomCommandLineQuery(string title, atom **_target, string _description) :
    Dialog::AtomQuery(title,_target, _description)
{}

CommandLineDialog::AtomCommandLineQuery::~AtomCommandLineQuery() {}

bool CommandLineDialog::AtomCommandLineQuery::handle() {
  int IdxOfAtom = -1;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    IdxOfAtom = CommandLineParser::getInstance().vm[getTitle()].as<int>();
    tmp = World::getInstance().getAtom(AtomById(IdxOfAtom));
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing atom for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::MoleculeCommandLineQuery::MoleculeCommandLineQuery(string title, molecule **_target, string _description) :
    Dialog::MoleculeQuery(title,_target, _description)
{}

CommandLineDialog::MoleculeCommandLineQuery::~MoleculeCommandLineQuery() {}

bool CommandLineDialog::MoleculeCommandLineQuery::handle() {
  int IdxOfMol = -1;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    IdxOfMol = CommandLineParser::getInstance().vm[getTitle()].as<int>();
    cout << "IdxOfMol " << IdxOfMol << endl;
    if (IdxOfMol >= 0)
      tmp = World::getInstance().getMolecule(MoleculeById(IdxOfMol));
    else
      tmp = NULL;
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing molecule for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::VectorCommandLineQuery::VectorCommandLineQuery(string title, Vector *_target, bool _check, string _description) :
    Dialog::VectorQuery(title,_target,_check, _description)
{}

CommandLineDialog::VectorCommandLineQuery::~VectorCommandLineQuery()
{}

bool CommandLineDialog::VectorCommandLineQuery::handle() {
  VectorValue temp;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    temp = CommandLineParser::getInstance().vm[getTitle()].as< VectorValue >();
    tmp->at(0) = temp.x;
    tmp->at(1) = temp.y;
    tmp->at(2) = temp.z;
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing vector for " << getTitle() << "." << endl);
    return false;
  }
}


CommandLineDialog::BoxCommandLineQuery::BoxCommandLineQuery(string title, Box* _cellSize, string _description) :
    Dialog::BoxQuery(title,_cellSize, _description)
{}

CommandLineDialog::BoxCommandLineQuery::~BoxCommandLineQuery()
{}

bool CommandLineDialog::BoxCommandLineQuery::handle() {
  BoxValue temp;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    temp = CommandLineParser::getInstance().vm[getTitle()].as< BoxValue >();
    tmp[0] = temp.xx;
    tmp[1] = temp.xy;
    tmp[2] = temp.xz;
    tmp[3] = temp.yy;
    tmp[4] = temp.yz;
    tmp[5] = temp.zz;
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing symmetric box matrix for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::ElementCommandLineQuery::ElementCommandLineQuery(string title, std::vector<element *> *target, string _description) :
    Dialog::ElementQuery(title,target, _description)
{}

CommandLineDialog::ElementCommandLineQuery::~ElementCommandLineQuery()
{}

bool CommandLineDialog::ElementCommandLineQuery::handle() {
  // TODO: vector of ints and removing first is not correctly implemented yet. How to remove from a vector?
  periodentafel *periode = World::getInstance().getPeriode();
  element *elemental = NULL;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    vector<int> AllElements = CommandLineParser::getInstance().vm[getTitle()].as< vector<int> >();
    for (vector<int>::iterator ZRunner = AllElements.begin(); ZRunner != AllElements.end(); ++ZRunner) {
      elemental = periode->FindElement(*ZRunner);
      ASSERT(elemental != NULL, "Invalid element specified in ElementCommandLineQuery");
      elements.push_back(elemental);
    }
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing element for " << getTitle() << "." << endl);
    return false;
  }
}
