/*
 * STLVectorValidator.hpp
 *
 *  Created on: May 9, 2012
 *      Author: heber
 */

#ifndef STLVECTORVALIDATOR_HPP_
#define STLVECTORVALIDATOR_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DummyValidator.hpp"
#include "RangeValidator.hpp"
#include "Validator.hpp"

#include <vector>

#include "CodePatterns/Range.hpp"

/** This class is a combination of \ref Validator<size_t> on the number of entries in
 * the vector while each element is checked with a given Validator<value_type>.
 */
template <class T>
class STLVectorValidator : public Validator< T >
{
  //!> this should throw an error at compile time if T is not a container
  typedef typename T::value_type value_type;

public:
  /** Constructor for class STLVectorValidator, number of elements given by a range.
   *
   * @param validator validator for the single element
   * @param range range of valid number of elements
   */
  STLVectorValidator(const range<size_t> &range, const Validator<value_type> &validator) :
    ElementwiseValidator(validator.clone()),
    NumberOfElementsValidator(new RangeValidator<size_t>(range))
  {}

  /** Constructor for class STLVectorValidator, number of elements given by a range.
   *
   * @param range range of valid number of elements
   */
  STLVectorValidator(const range<size_t> &range) :
    ElementwiseValidator(new DummyValidator<value_type>()),
    NumberOfElementsValidator(new RangeValidator<size_t>(range))
  {}

  /** Constructor for class STLVectorValidator, number of elements given by interval [\a min,\a max).
   *
   * @param validator validator for the single element
   * @param min lower bound of interval for valid number of elements
   * @param max upper bound of interval for valid number of elements
   */
  STLVectorValidator(const size_t min, const size_t max, const Validator<value_type> &validator) :
    ElementwiseValidator(validator.clone()),
    NumberOfElementsValidator(new RangeValidator<size_t>(min,max))
  {}

  /** Constructor for class STLVectorValidator, number of elements given by interval [\a min,\a max).
   *
   * @param min lower bound of interval for valid number of elements
   * @param max upper bound of interval for valid number of elements
   */
  STLVectorValidator(const size_t min, const size_t max) :
    ElementwiseValidator(new DummyValidator<value_type>()),
    NumberOfElementsValidator(new RangeValidator<size_t>(min,max))
  {}

  /** Constructor for class STLVectorValidator, any number of elements is allowed.
   *
   * @param validator validator for the single element
   */
  STLVectorValidator(const Validator<value_type> &validator) :
    ElementwiseValidator(validator.clone()),
    NumberOfElementsValidator(new DummyValidator<size_t>())
  {}

  /** Constructor for class STLVectorValidator, any number of unrestricted elements is allowed.
   *
   */
  STLVectorValidator() :
    ElementwiseValidator(new DummyValidator<value_type>()),
    NumberOfElementsValidator(new DummyValidator<size_t>())
  {}

  /** Destructor for class STLVectorValidator
   *
   */
  ~STLVectorValidator()
  {
    delete ElementwiseValidator;
    delete NumberOfElementsValidator;
  }

  bool isValid(const T & _value) const
  {
    bool status = NumberOfElementsValidator->isValid(_value.size());
    for (typename std::vector<value_type>::const_iterator iter = _value.begin();
        iter != _value.end(); ++iter) {
      status = status && ElementwiseValidator->isValid(*iter);
    }
    return status;
  }
  bool operator==(const Validator<T> &_instance) const
  {
    const STLVectorValidator<T> *inst = dynamic_cast<const STLVectorValidator<T> *>(&_instance);
    bool status = *NumberOfElementsValidator == *inst->NumberOfElementsValidator;
    status = status && *ElementwiseValidator == *inst->ElementwiseValidator;
    return status;
  }

  Validator<T>* clone() const
  {
    Validator<T> *inst = new STLVectorValidator<T>(*NumberOfElementsValidator, *ElementwiseValidator);
    return inst;
  }

private:
  /** Constructor for class STLVectorValidator, number of elements given by \ref RangeValidator.
   *
   * \note Is only used in STLVectorValidator::clone() as it may lead to ambiguities
   *  with a signature of two validators in case value_type equals size_t.
   *
   * @param validator validator for the single element
   * @param rangevalidator validator for the valid number of elements
   */
  STLVectorValidator(const Validator<size_t> &rangevalidator, const Validator<value_type> &validator) :
    ElementwiseValidator(validator.clone()),
    NumberOfElementsValidator(dynamic_cast< Validator<size_t> *>(rangevalidator.clone()))
  {}


private:
  Validator<value_type> *ElementwiseValidator;
  Validator<size_t> *NumberOfElementsValidator;
};


#endif /* STLVECTORVALIDATOR_HPP_ */
