/*
 * RandomNumberEngine_Encapsulation.hpp
 *
 *  Created on: Jan 01, 2011
 *      Author: heber
 */

#ifndef RANDOMNUMBERENGINE_ENCAPSULATION_HPP_
#define RANDOMNUMBERENGINE_ENCAPSULATION_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <typeinfo>

#include <boost/nondet_random.hpp>
#include <boost/random.hpp>
#include <boost/random/additive_combine.hpp>
#include <boost/random/discard_block.hpp>
#include <boost/random/inversive_congruential.hpp>
#include <boost/random/lagged_fibonacci.hpp>
#include <boost/random/linear_congruential.hpp>
#include <boost/random/linear_feedback_shift.hpp>
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/random_number_generator.hpp>
#include <boost/random/ranlux.hpp>
#include <boost/random/shuffle_output.hpp>
#include <boost/random/subtract_with_carry.hpp>
#include <boost/random/xor_combine.hpp>

#include "CodePatterns/ManipulableClone.hpp"
#include "RandomNumberEngine.hpp"

class RandomNumberEngineFactory;

#include "RandomNumberEngine_Parameters.hpp"

/** Template class that encapsulates the random number engines from
 *  random::boost.
 *
 *
 * We need one template parameters:
 * -# the engine - generates uniform random numbers
 */
template <class engine>
class RandomNumberEngine_Encapsulation :
  public RandomNumberEngine,
  public ManipulableClone<RandomNumberEngine, RandomNumberEngine_Parameters>
{
    /**
     * Factory is friend such that it can access private cstor when filling its
     * table
     */
    friend class RandomNumberEngineFactory;

public:
  /** Getter for the whole set of possible parameters.
   *
   * @return filled instance of RandomNumberEngine_Parameters
   */
    RandomNumberEngine_Parameters* getParameterSet() const
  {
    RandomNumberEngine_Parameters *params = new RandomNumberEngine_Parameters();
    params->getParameters(this);
    return params;
  }

  /** Set the engine's seed.
   *
   * @param _seed seed to set to
   */
  void seed(unsigned int _seed) {
    value_seed = _seed;
    engine_type.seed(value_seed);
  }

  /** Getter for the engine's seed.
   *
   * @return seed
   */
  unsigned int getseed() const {
    return value_seed;
  }

  /** Getter for smallest value the engine produces.
   *
   * @return smallest value
   */
  double min() const {
    return engine_type.min();
  }

  /** Getter for largest value the engine produces.
   *
   * @return largest value
   */
  double max() const {
    return engine_type.max();
  }

  /** Getter for the type name of the internal engine.
   *
   */
  std::string name() {
    return typeid(engine_type).name();
  }

  /** Getter for the engine instance.
   *
   * @return reference to instance
   */
  engine& getEngine() {
    return engine_type;
  }

  /** Clones the current instance.
   *
   * Implementation of Clone pattern.
   *
   * @return interface reference to cloned instance
   */
  RandomNumberEngine* clone() const {
    RandomNumberEngine_Parameters *params = getParameterSet();
    RandomNumberEngine* MyClone = new RandomNumberEngine_Encapsulation<engine>(*params);
    delete params;
    //MyClone->seed(getseed());
    return MyClone;
  }

  /** Clones and manipulates the current instance.
   *
   * Implementation of ManipulableClone pattern.
   *
   * @param _params set of parameters to instantiate manipulated clone with
   * @return interface reference to cloned&manipulated instance
   */
  ManipulableClone<RandomNumberEngine, RandomNumberEngine_Parameters>*
  manipulatedclone(const RandomNumberEngine_Parameters &_params) const {
    RandomNumberEngine_Encapsulation<engine>* MyClone = NULL;
    MyClone = new RandomNumberEngine_Encapsulation<engine>();
    MyClone->value_seed = _params.seed;
    MyClone->seed(_params.seed);
    return MyClone;
  }

protected:
  /** Constructor that instantiates a specific random number generator and
   * distribution.
   * @param _engine_type instance of the desired engine
   */
  RandomNumberEngine_Encapsulation() :
    value_seed(1)
  {}

  /** Constructor that instantiates a specific random number generator and
   * distribution.
   * @param _engine_type instance of the desired engine
   */
  RandomNumberEngine_Encapsulation(const RandomNumberEngine_Parameters &_params) :
    value_seed(_params.seed)
  {}

  /** Destructor of the class.
   *
   */
  virtual ~RandomNumberEngine_Encapsulation() {}
private:
  engine engine_type;
  unsigned int value_seed;
};

#endif /* RANDOMNUMBERENGINE_ENCAPSULATION_HPP_ */
