/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ParsePotentialsAction.cpp
 *
 *  Created on: Sep 12, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// and before MemDebug due to placement new
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

//#include "CodePatterns/MemDebug.hpp"

#include "Actions/PotentialAction/ParsePotentialsAction.hpp"

#include <boost/filesystem.hpp>
#include <fstream>

#include "CodePatterns/Log.hpp"

#include "Potentials/EmpiricalPotential.hpp"
#include "Potentials/Exceptions.hpp"
#include "Potentials/PotentialFactory.hpp"
#include "Potentials/PotentialRegistry.hpp"
#include "Potentials/RegistryDeserializer.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "ParsePotentialsAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

ActionState::ptr PotentialParsePotentialsAction::performCall()
{
  if (boost::filesystem::exists(params.potential_file.get())) {
    std::ifstream returnstream(params.potential_file.get().string().c_str());
    if (returnstream.good()) {
      try {
        RegistryDeserializer<EmpiricalPotential> deserialize(
            returnstream,
            PotentialFactory::getInstance(),
            PotentialRegistry::getInstance(),
            std::string("potential"));
        deserialize();
      } catch (SerializerMissingValueException &e) {
        if (const std::string *key = boost::get_error_info<SerializerKey>(e)) {
          STATUS("Missing value when parsing information for potential "+*key+".");
        } else
          STATUS("Missing value parsing information for potential with unknown key.");
        return Action::failure;
      } catch (SerializerIllegalKeyException &e) {
        if (const std::string *key = boost::get_error_info<SerializerKey>(e)) {
          STATUS("Illegal key parsing information for potential "+*key+".");
        } else {
          STATUS("Illegal key parsing information for potential with unknown key.");
        }
        return Action::failure;
      }
    } else {
      STATUS("Failed to parse from "+params.potential_file.get().string()+".");
      return Action::failure;
    }
    returnstream.close();

    LOG(0, "STATUS: Parsed a set of potentials parsed from "
        << params.potential_file.get().string() << ".");

  } else {
      STATUS("No existing potential file given!");
      return Action::failure;
  }

  return Action::success;
}

ActionState::ptr PotentialParsePotentialsAction::performUndo(ActionState::ptr _state) {
  STATUS("Undo of PotentialParsePotentialsAction not implemented.");
  return Action::failure;
}

ActionState::ptr PotentialParsePotentialsAction::performRedo(ActionState::ptr _state){
  STATUS("Redo of PotentialParsePotentialsAction not implemented.");
  return Action::failure;
}

bool PotentialParsePotentialsAction::canUndo() {
  return false;
}

bool PotentialParsePotentialsAction::shouldUndo() {
  return false;
}
/** =========== end of function ====================== */
