/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * HomologyGraphUnitTest.cpp
 *
 *  Created on: Sep 25, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "Fragmentation/Homology/HomologyGraph.hpp"
#include "Fragmentation/Graph.hpp"

#include "HomologyGraphUnitTest.hpp"

#include "CodePatterns/Assert.hpp"

#include <boost/assign.hpp>

using namespace boost::assign;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( HomologyGraphTest );


void HomologyGraphTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // add nodes
  nodes += FragmentNode(1,1),FragmentNode(1,4), FragmentNode(2,2), FragmentNode(2,4);
  othernodes += FragmentNode(1,1),FragmentNode(1,4);

  // add edges
  edges += FragmentEdge(1,1),FragmentEdge(1,4), FragmentEdge(2,2), FragmentEdge(2,4);
  otheredges += FragmentEdge(1,4),FragmentEdge(2,2);

  // construct graph
  graph = new HomologyGraph(nodes, edges);
}


void HomologyGraphTest::tearDown()
{
  delete graph;
}

/** UnitTest for whether homologies are correctly recognized
 */
void HomologyGraphTest::comparatorTest()
{
  // construct other graphs
  HomologyGraph samegraph(nodes, edges);
  HomologyGraph othergraph(nodes, otheredges);
  HomologyGraph anothergraph(othernodes, edges);
  HomologyGraph yetanothergraph(othernodes, otheredges);

  // sort them (order is: anothergraph, yetanothergraph, graph/samegraph, othergraph)
  CPPUNIT_ASSERT( !(*graph < samegraph) );
  CPPUNIT_ASSERT( othergraph > *graph );
  CPPUNIT_ASSERT( anothergraph < *graph );
  CPPUNIT_ASSERT( yetanothergraph < *graph );
  CPPUNIT_ASSERT( anothergraph < othergraph );
  CPPUNIT_ASSERT( yetanothergraph < othergraph );
  CPPUNIT_ASSERT( yetanothergraph > anothergraph );

  CPPUNIT_ASSERT( !(*graph > samegraph) );
  CPPUNIT_ASSERT( *graph < othergraph );
  CPPUNIT_ASSERT( *graph > anothergraph );
  CPPUNIT_ASSERT( *graph > yetanothergraph );
  CPPUNIT_ASSERT( othergraph > anothergraph );
  CPPUNIT_ASSERT( othergraph > yetanothergraph );
  CPPUNIT_ASSERT( anothergraph < yetanothergraph );

  // compare them
  CPPUNIT_ASSERT( *graph == samegraph );
  CPPUNIT_ASSERT( *graph != othergraph );
  CPPUNIT_ASSERT( *graph != anothergraph );
  CPPUNIT_ASSERT( *graph != yetanothergraph );
  CPPUNIT_ASSERT( othergraph != anothergraph );
  CPPUNIT_ASSERT( othergraph != yetanothergraph );
  CPPUNIT_ASSERT( anothergraph != yetanothergraph );
}

/** UnitTest for operator=()
 */
void HomologyGraphTest::assigmentTest()
{
  HomologyGraph samegraph(nodes, edges);
  HomologyGraph othergraph(othernodes, otheredges);

  // check initial status
  CPPUNIT_ASSERT( *graph == samegraph);
  CPPUNIT_ASSERT( *graph != othergraph);

  // now assign
  samegraph = othergraph;
  othergraph = *graph;

  // and check again
  CPPUNIT_ASSERT( *graph != samegraph);
  CPPUNIT_ASSERT( *graph == othergraph);

  // also self-assign
  *graph = *graph;
  CPPUNIT_ASSERT( *graph == *graph);
}
