/*
 * HomologyGraph.hpp
 *
 *  Created on: Sep 24, 2012
 *      Author: heber
 */

#ifndef HOMOLOGYGRAPH_HPP_
#define HOMOLOGYGRAPH_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <set>

#include "Fragmentation/Homology/FragmentEdge.hpp"
#include "Fragmentation/Homology/FragmentNode.hpp"

/** This class contains the representation of a molecular fragment as a graph.
 *
 * Only, we do not store the full graph in here. We have to include symmetries
 * such that two hydrogens may switch places. Eventually, we only look for the
 * set of distances of a fragment. If two hydrogens switch places, then also in
 * the set of distances some distances are interchanged but the whole fragment
 * remains the same. Hence, we have to store the bond graph representation in
 * such a way as to automatically include these symmetries.
 *
 * To this end, we use FragmentNode and FragmentEdge to store the vital
 * information.
 *
 */
class HomologyGraph
{
public:
  //!> typedef for a set of nodes representing node information
  typedef std::set<FragmentNode> nodes_t;
  //!> typedef for a set of nodes representing edge information
  typedef std::set<FragmentEdge> edges_t;
public:
  /** Default constructor for class HomologyGraph.
   *
   * This is required to allow placement in STL containers
   *
   */
  HomologyGraph() {}

  /** Constructor for class HomologyGraph.
   *
   * @param _nodes information on nodes of this graph
   * @param _edges information on edges of this graph
   */
  HomologyGraph(const nodes_t &_nodes, const edges_t &_edges) :
    nodes(_nodes),
    edges(_edges)
  {}

  /** Destructor for class HomologyGraph.
   *
   */
  ~HomologyGraph() {}

  // comparators (allows sorting and hence quicker finding in STL containers)
  bool operator<(const HomologyGraph &graph) const;
  bool operator>(const HomologyGraph &graph) const;
  bool operator==(const HomologyGraph &graph) const;
  bool operator!=(const HomologyGraph &graph) const {
    return (!(*this == graph));
  }

  /** Assignment operator for class HomologyGraph.
   *
   * This is required to allow placement in STL container as we need to
   * const_cast override our const member variables.
   *
   */
  HomologyGraph& operator=(const HomologyGraph &graph);

private:
  //!> information on the nodes of the graph
  const nodes_t nodes;
  //!> information on the edges of the graph
  const edges_t edges;
};


#endif /* HOMOLOGYGRAPH_HPP_ */
