/*
 * FunctionModel.hpp
 *
 *  Created on: 02.10.2012
 *      Author: heber
 */

#ifndef FUNCTIONMODEL_HPP_
#define FUNCTIONMODEL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/function.hpp>
#include <vector>

#include "FunctionApproximation/FunctionArgument.hpp"

class Fragment;

/** This class represents the interface for a given function to model a
 * high-dimensional data set in FunctionApproximation.
 *
 * As the parameters may be stored differently, the interface functions for
 * getting and setting them are as light-weight (and not speed-optimized)
 * as possible.
 *
 */
class FunctionModel
{
public:
  //!> typedef for a single parameter degree of freedom of the function
  typedef double parameter_t;
  //!> typedef for the whole set of parameters of the function
  typedef std::vector<parameter_t> parameters_t;
  //!> typedef for the argument vector as input to the function
  typedef std::vector<argument_t> arguments_t;
  //!> typedef for a single result degree of freedom
  typedef double result_t;
  //!> typedef for the result vector as returned by the function
  typedef std::vector<result_t> results_t;
  //!> typedef for a function containing how to extract required information from a Fragment.
  typedef boost::function< arguments_t (const Fragment &, const size_t)> extractor_t;
  //!> typedef for a vector of particle tyoes
  typedef std::vector< double > charges_t;

public:
  FunctionModel() {}
  virtual ~FunctionModel() {}

  /** Setter for the parameters of the model function.
   *
   * \param params set of parameters to set
   */
  virtual void setParameters(const parameters_t &params)=0;

  /** Getter for the parameters of this model function.
   *
   * \return current set of parameters of the model function
   */
  virtual parameters_t getParameters() const=0;

//  /** Sets the parameter randomly within the sensible range of each parameter.
//   *
//   */
//  virtual void setParametersToRandomInitialValues() const=0;

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  virtual size_t getParameterDimension() const=0;

  /** Evaluates the function with the given \a arguments and the current set of
   * parameters.
   *
   * \param arguments set of arguments as input variables to the function
   * \return result of the function
   */
  virtual results_t operator()(const arguments_t &arguments) const=0;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param arguments set of arguments as input variables to the function
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  virtual results_t parameter_derivative(const arguments_t &arguments, const size_t index) const=0;

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  virtual bool isBoxConstraint() const=0;

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  virtual parameters_t getLowerBoxConstraints() const=0;

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  virtual parameters_t getUpperBoxConstraints() const=0;

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * \param charges vector of charges to be extracted
   * \return bound function extracting distances from a fragment
   */
  virtual extractor_t getFragmentSpecificExtractor(const charges_t &charges) const=0;
};

#endif /* FUNCTIONMODEL_HPP_ */
