/** \file periodentafel.cpp
 *
 * Function implementations for the class periodentafel.
 *
 */

using namespace std;

#include <iomanip>
#include <fstream>
#include <cstring>
#include <cassert>

#include "element.hpp"
#include "helpers.hpp"
#include "lists.hpp"
#include "log.hpp"
#include "periodentafel.hpp"
#include "verbose.hpp"

using namespace std;

/************************************* Functions for class periodentafel ***************************/

/** constructor for class periodentafel
 * Initialises start and end of list and resets periodentafel::checkliste to false.
 */
periodentafel::periodentafel()
{};

/** destructor for class periodentafel
 * Removes every element and afterwards deletes start and end of list.
 */
periodentafel::~periodentafel()
{
  CleanupPeriodtable();
};

/** Adds element to period table list
 * \param *pointer element to be added
 * \return true - succeeded, false - does not occur
 */
periodentafel::iterator periodentafel::AddElement(element * const pointer)
{
  atomicNumber_t Z = pointer->getNumber();
  assert(!elements.count(Z));
  pointer->sort = &pointer->Z;
  if (pointer->getNumber() < 1 && pointer->getNumber() >= MAX_ELEMENTS)
    Log() << Verbose(0) << "Invalid Z number!\n";
  pair<iterator,bool> res = elements.insert(pair<atomicNumber_t,element*>(Z,pointer));
  return res.first;
};

/** Removes element from list.
 * \param *pointer element to be removed
 * \return true - succeeded, false - element not found
 */
void periodentafel::RemoveElement(element * const pointer)
{
  atomicNumber_t Z = pointer->getNumber();
  elements.erase(Z);
};

/** Removes every element from the period table.
 * \return true - succeeded, false - does not occur
 */
void periodentafel::CleanupPeriodtable()
{
  for(iterator iter=elements.begin();iter!=elements.end();++iter){
    delete(*iter).second;
  }
  elements.clear();
};

/** Finds an element by its atomic number.
 * If element is not yet in list, returns NULL.
 * \param Z atomic number
 * \return pointer to element or NULL if not found
 */
const element * periodentafel::FindElement(atomicNumber_t Z) const
{
  const_iterator res = elements.find(Z);
  return res!=elements.end()?((*res).second):0;
};

/** Finds an element by its atomic number.
 * If element is not yet in list, datas are asked and stored in database.
 * \param shorthand chemical symbol of the element, e.g. H for hydrogene
 * \return pointer to element
 */
const element * periodentafel::FindElement(const char * const shorthand) const
{
  element *res = 0;
  for(const_iterator iter=elements.begin();iter!=elements.end();++iter) {
    if((*iter).second->getSymbol() == shorthand){
      res = (*iter).second;
      break;
    }
  }
  return res;
};

/** Asks for element number and returns pointer to element
 */
const element * periodentafel::AskElement() const
{
  const element *walker = NULL;
  int Z;
  do {
    Log() << Verbose(0) << "Atomic number Z: ";
    cin >> Z;
    walker = this->FindElement(Z);  // give type
  } while (walker == NULL);
  return walker;
};

/** Asks for element and if not found, presents mask to enter info.
 * \return pointer to either present or newly created element
 */
const element * periodentafel::EnterElement()
{
  const element *res = NULL;
  atomicNumber_t Z = 0;
  Log() << Verbose(0) << "Atomic number: " << Z << endl;
  cin >> Z;
  res = FindElement(Z);
  if (!res) {
    // TODO: make this using the constructor
    element *tmp;
    Log() << Verbose(0) << "Element not found in database, please enter." << endl;
    tmp = new element;
    tmp->Z = Z;
    Log() << Verbose(0) << "Mass: " << endl;
    cin >> tmp->mass;
    Log() << Verbose(0) << "Name [max 64 chars]: " << endl;
    cin >> tmp->name;
    Log() << Verbose(0) << "Short form [max 3 chars]: " << endl;
    cin >> tmp->symbol;
    AddElement(tmp);
    res = tmp;
  }
  return res;
};


/******************** Access to iterators ****************************/
periodentafel::const_iterator periodentafel::begin(){
  return elements.begin();
}

periodentafel::const_iterator periodentafel::end(){
  return elements.end();
}

periodentafel::reverse_iterator periodentafel::rbegin(){
  return reverse_iterator(elements.end());
}

periodentafel::reverse_iterator periodentafel::rend(){
  return reverse_iterator(elements.begin());
}

/** Prints period table to given stream.
 * \param output stream
 */
bool periodentafel::Output(ostream * const output) const
{
  bool result = true;
  if (output != NULL) {
    for(const_iterator iter=elements.begin(); iter !=elements.end();++iter){
      result = result && (*iter).second->Output(output);
    }
    return result;
  } else
    return false;
};

/** Prints period table to given stream.
 * \param *output output stream
 * \param *checkliste elements table for this molecule
 */
bool periodentafel::Checkout(ostream * const output, const int * const checkliste) const
{
  bool result = true;
  int No = 1;

  if (output != NULL) {
    *output << "# Ion type data (PP = PseudoPotential, Z = atomic number)" << endl;
    *output << "#Ion_TypeNr.\tAmount\tZ\tRGauss\tL_Max(PP)L_Loc(PP)IonMass\t# chemical name, symbol" << endl;
    for(const_iterator iter=elements.begin(); iter!=elements.end();++iter){
      if (((*iter).first < MAX_ELEMENTS) && (checkliste[(*iter).first])) {
        (*iter).second->No = No;
        result = result && (*iter).second->Checkout(output, No++, checkliste[(*iter).first]);
      }
    }
    return result;
  } else
    return false;
};

/** Loads element list from file.
 * \param *path to to standard file names
 */
bool periodentafel::LoadPeriodentafel(const char *path)
{
  ifstream infile;
  element *ptr;
  map<atomicNumber_t,element*> parsedElems;
  bool status = true;
  bool otherstatus = true;
  char filename[255];

  // fill elements DB
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDELEMENTSDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    infile.getline(header1, MAXSTRINGSIZE);
    infile.getline(header2, MAXSTRINGSIZE); // skip first two header lines
    Log() << Verbose(0) <<  "Parsed elements:";
    while (!infile.eof()) {
      element *neues = new element;
      infile >> neues->name;
      //infile >> ws;
      infile >> neues->symbol;
      //infile >> ws;
      infile >> neues->period;
      //infile >> ws;
      infile >> neues->group;
      //infile >> ws;
      infile >> neues->block;
      //infile >> ws;
      infile >> neues->Z;
      //infile >> ws;
      infile >> neues->mass;
      //infile >> ws;
      infile >> neues->CovalentRadius;
      //infile >> ws;
      infile >> neues->VanDerWaalsRadius;
      //infile >> ws;
      infile >> ws;
      Log() << Verbose(0) << " " << neues->symbol;
      //neues->Output((ofstream *)&cout);
      if ((neues->Z > 0) && (neues->Z < MAX_ELEMENTS))
        parsedElems[neues->getNumber()] = neues;
      else {
        Log() << Verbose(0) << "Could not parse element: ";
        neues->Output((ofstream *)&cout);
        delete(neues);
      }
    }
    Log() << Verbose(0) << endl;
    infile.close();
    infile.clear();
  } else
    status = false;

  // fill valence DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDVALENCEDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
      atomicNumber_t Z;
      infile >> Z;
      infile >> ws;
      infile >> parsedElems[Z]->Valence;
      infile >> ws;
      //Log() << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->Valence << " valence electrons." << endl;
    }
    infile.close();
    infile.clear();
  } else
    otherstatus = false;

  // fill valence DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDORBITALDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
      atomicNumber_t Z;
      infile >> Z;
      infile >> ws;
      infile >> parsedElems[Z]->NoValenceOrbitals;
      infile >> ws;
      //Log() << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->NoValenceOrbitals << " number of singly occupied valence orbitals." << endl;
    }
    infile.close();
    infile.clear();
  } else
    otherstatus = false;

  // fill H-BondDistance DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDHBONDDISTANCEDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
      atomicNumber_t Z;
      infile >> Z;
      ptr = parsedElems[Z];
      infile >> ws;
      infile >> ptr->HBondDistance[0];
      infile >> ptr->HBondDistance[1];
      infile >> ptr->HBondDistance[2];
      infile >> ws;
      //Log() << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->HBondDistance[0] << " Angstrom typical distance to hydrogen." << endl;
    }
    infile.close();
    infile.clear();
  } else
    otherstatus = false;

  // fill H-BondAngle DB per element
  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDHBONDANGLEDB, MAXSTRINGSIZE-strlen(filename));
  infile.open(filename);
  if (infile != NULL) {
    while (!infile.eof()) {
      atomicNumber_t Z;
      infile >> Z;
      ptr = parsedElems[Z];
      infile >> ws;
      infile >> ptr->HBondAngle[0];
      infile >> ptr->HBondAngle[1];
      infile >> ptr->HBondAngle[2];
      infile >> ws;
      //Log() << Verbose(3) << "Element " << (int)tmp << " has " << FindElement((int)tmp)->HBondAngle[0] << ", " << FindElement((int)tmp)->HBondAngle[1] << ", " << FindElement((int)tmp)->HBondAngle[2] << " degrees bond angle for one, two, three connected hydrogens." << endl;
    }
    infile.close();
  } else
    otherstatus = false;

  if (otherstatus){
    map<atomicNumber_t,element*>::iterator iter;
    for(iter=parsedElems.begin();iter!=parsedElems.end();++iter){
      AddElement((*iter).second);
    }
  }
  else{
    eLog() << Verbose(2) << "Something went wrong while parsing the other databases!" << endl;
    map<atomicNumber_t,element*>::iterator iter;
    for(iter=parsedElems.begin();iter!=parsedElems.end();++iter){
      AddElement((*iter).second);
    }
  }

  return status;
};

/** Stores element list to file.
 */
bool periodentafel::StorePeriodentafel(const char *path) const
{
  bool result = true;
  ofstream f;
  char filename[MAXSTRINGSIZE];

  strncpy(filename, path, MAXSTRINGSIZE);
  strncat(filename, "/", MAXSTRINGSIZE-strlen(filename));
  strncat(filename, STANDARDELEMENTSDB, MAXSTRINGSIZE-strlen(filename));
  f.open(filename);
  if (f != NULL) {
    f << header1 << endl;
    f << header2 << endl;
    for(const_iterator iter=elements.begin();iter!=elements.end();++iter){
         result = result && (*iter).second->Output(&f);
    }
    f.close();
  } else
    result = false;
  return result;
};
