/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2017 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BreadthFirstSearchGathererUnitTest.cpp
 *
 *  Created on: May 19, 2017
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <boost/assign.hpp>

#include "CodePatterns/Log.hpp"

#include "Graph/BoostGraphCreator.hpp"
#include "Graph/BreadthFirstSearchGatherer.hpp"

#include "BreadthFirstSearchGathererUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( BreadthFirstSearchGathererTest );

typedef std::pair<int,int> E;

void BreadthFirstSearchGathererTest::setUp()
{
  BGCreator = new BoostGraphCreator();
};


void BreadthFirstSearchGathererTest::tearDown()
{
  delete BGCreator;
};

/** Little helper function to prepare a linear graph with vertices
 * correctly named (not just indexed) and also the internal node
 * map prepared.
 */
void BreadthFirstSearchGathererTest::prepareLinearGraph()
{
  E edges[] = { E(0,1), E(1,2), E(2,3), E(3,4) };
  const size_t no_nodes = 5;
  BGCreator->graph =
      BoostGraphCreator::UndirectedGraph(edges, edges + sizeof(edges) / sizeof(E), no_nodes);
  BGCreator->atomids_nodeids +=
      make_pair(0,0), make_pair(1,1), make_pair(2,2), make_pair(3,3), make_pair(4,4);
  for (size_t i=0;i<no_nodes;++i)
    boost::put(boost::get(boost::vertex_name, BGCreator->graph), boost::vertex(i, BGCreator->graph), i+1);
}

/** Tests whether operator() works.
 */
void BreadthFirstSearchGathererTest::operatorTest()
{
  // create linear test graph
  prepareLinearGraph();

  // call operator with default argument
  BreadthFirstSearchGatherer gatherer(*BGCreator);
  std::vector<atomId_t> atomids = gatherer(0);

  // create comparator set
  std::vector<atomId_t> compareids;
  compareids += 1,2,3,4,5;
  CPPUNIT_ASSERT_EQUAL ((size_t)5, atomids.size());
  CPPUNIT_ASSERT_EQUAL (compareids, atomids);
};
