/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Tesselation_InsideOutsideUnitTest.cpp
 *
 *  Created on: Dec 28, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>
#include <iostream>

#include "Atom/TesselPoint.hpp"
#include "CodePatterns/Log.hpp"
#include "Helpers/defs.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "Tesselation/BoundaryLineSet.hpp"
#include "Tesselation/BoundaryTriangleSet.hpp"
#include "Tesselation/CandidateForTesselation.hpp"
#include "Tesselation/tesselation.hpp"

#include "Tesselation_InsideOutsideUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

const double TesselationInOutsideTest::SPHERERADIUS=2.;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( TesselationInOutsideTest );

void TesselationInOutsideTest::setUp()
{
  setVerbosity(2);

  // create corners
  class TesselPoint *Walker;
  Walker = new TesselPoint;
  Walker->setPosition(Vector(0., 0., 0.));
  Walker->setName("1");
  Walker->setNr(1);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(0., 1., 0.));
  Walker->setName("2");
  Walker->setNr(2);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(1., 0., 0.));
  Walker->setName("3");
  Walker->setNr(3);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(1., 1., 0.));
  Walker->setName("4");
  Walker->setNr(4);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(0., 0., 1.));
  Walker->setName("5");
  Walker->setNr(5);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(0., 1., 1.));
  Walker->setName("6");
  Walker->setNr(6);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(1., 0., 1.));
  Walker->setName("7");
  Walker->setNr(7);
  Corners.push_back(Walker);
  Walker = new TesselPoint;
  Walker->setPosition(Vector(1., 1., 1.));
  Walker->setName("8");
  Walker->setNr(8);
  Corners.push_back(Walker);

  // create linkedcell
  PointCloudAdaptor< TesselPointSTLList > cloud(&Corners, "TesselPointSTLList");
  LinkedList = new LinkedCell_deprecated(cloud, 2.*SPHERERADIUS);

  // create tesselation
  TesselStruct = new Tesselation;
  TesselStruct->FindStartingTriangle(SPHERERADIUS, LinkedList);

  CandidateForTesselation *baseline = NULL;
  BoundaryTriangleSet *T = NULL;
  bool OneLoopWithoutSuccessFlag = true;
  bool TesselationFailFlag = false;
  while ((!TesselStruct->OpenLines.empty()) && (OneLoopWithoutSuccessFlag)) {
    // 2a. fill all new OpenLines
    LOG(1, "There are " << TesselStruct->OpenLines.size() << " open lines to scan for candidates:");
    for (CandidateMap::iterator Runner = TesselStruct->OpenLines.begin(); Runner != TesselStruct->OpenLines.end(); Runner++)
      LOG(2, *(Runner->second));

    for (CandidateMap::iterator Runner = TesselStruct->OpenLines.begin(); Runner != TesselStruct->OpenLines.end(); Runner++) {
      baseline = Runner->second;
      if (baseline->pointlist.empty()) {
        T = (((baseline->BaseLine->triangles.begin()))->second);
        LOG(1, "Finding best candidate for open line " << *baseline->BaseLine << " of triangle " << *T);
        TesselationFailFlag = TesselStruct->FindNextSuitableTriangle(*baseline, *T, SPHERERADIUS, LinkedList); //the line is there, so there is a triangle, but only one.
      }
    }

    // 2b. search for smallest ShortestAngle among all candidates
    double ShortestAngle = 4.*M_PI;
    LOG(1, "There are " << TesselStruct->OpenLines.size() << " open lines to scan for the best candidates:");
    for (CandidateMap::iterator Runner = TesselStruct->OpenLines.begin(); Runner != TesselStruct->OpenLines.end(); Runner++)
      LOG(2, *(Runner->second));

    for (CandidateMap::iterator Runner = TesselStruct->OpenLines.begin(); Runner != TesselStruct->OpenLines.end(); Runner++) {
      if (Runner->second->ShortestAngle < ShortestAngle) {
        baseline = Runner->second;
        ShortestAngle = baseline->ShortestAngle;
        //LOG(1, "New best candidate is " << *baseline->BaseLine << " with point " << *baseline->point << " and angle " << baseline->ShortestAngle);
      }
    }
    if ((ShortestAngle == 4.*M_PI) || (baseline->pointlist.empty()))
      OneLoopWithoutSuccessFlag = false;
    else {
      TesselStruct->AddCandidatePolygon(*baseline, SPHERERADIUS, LinkedList);
    }
  }
};


void TesselationInOutsideTest::tearDown()
{
  delete(LinkedList);
  delete(TesselStruct);
  for (TesselPointSTLList::iterator Runner = Corners.begin(); Runner != Corners.end(); Runner++)
    delete(*Runner);
  Corners.clear();
  logger::purgeInstance();
  errorLogger::purgeInstance();
};

/** UnitTest for Tesselation::IsInnerPoint()
 */
void TesselationInOutsideTest::IsInnerPointTest()
{
  double n[3];
  const double boundary = 2.;
  const double step = 1.;

  // go through the mesh and check each point
  for (n[0] = -boundary; n[0] <= boundary; n[0]+=step)
    for (n[1] = -boundary; n[1] <= boundary; n[1]+=step)
      for (n[2] = -boundary; n[2] <= boundary; n[2]+=step) {
        if ( ((n[0] >= 0.) && (n[1] >= 0.) && (n[2] >= 0.)) && ((n[0] <= 1.) && (n[1] <= 1.) && (n[2] <= 1.)))
          CPPUNIT_ASSERT_EQUAL( true , TesselStruct->IsInnerPoint(Vector(n[0], n[1], n[2]), LinkedList) );
        else
          CPPUNIT_ASSERT_EQUAL( false , TesselStruct->IsInnerPoint(Vector(n[0], n[1], n[2]), LinkedList) );
      }
};
