/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FragmentEdgeUnitTest.cpp
 *
 *  Created on: Sep 24, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "Fragmentation/Homology/FragmentEdge.hpp"

#include "FragmentEdgeUnitTest.hpp"

#include "CodePatterns/Assert.hpp"

#include <sstream>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( FragmentEdgeTest );


void FragmentEdgeTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);
}


void FragmentEdgeTest::tearDown()
{
}

/** UnitTest for operator<(), operator>() and operator==()
 */
void FragmentEdgeTest::comparatorTest()
{
  FragmentEdge edge(1,1);
  FragmentEdge sameedge(1,1);
  FragmentEdge otheredge(1,4);
  FragmentEdge anotheredge(2,2);
  FragmentEdge yetanotheredge(2,4);

  CPPUNIT_ASSERT( !(edge < sameedge) );
  CPPUNIT_ASSERT( edge < otheredge );
  CPPUNIT_ASSERT( edge < anotheredge );
  CPPUNIT_ASSERT( edge < yetanotheredge );
  CPPUNIT_ASSERT( otheredge < anotheredge );
  CPPUNIT_ASSERT( otheredge < yetanotheredge );
  CPPUNIT_ASSERT( anotheredge < yetanotheredge );

  CPPUNIT_ASSERT( !(edge > sameedge) );
  CPPUNIT_ASSERT( otheredge > edge );
  CPPUNIT_ASSERT( anotheredge > edge );
  CPPUNIT_ASSERT( anotheredge > otheredge );
  CPPUNIT_ASSERT( yetanotheredge > edge );
  CPPUNIT_ASSERT( yetanotheredge > otheredge );
  CPPUNIT_ASSERT( yetanotheredge > anotheredge );

  CPPUNIT_ASSERT( edge == edge );
  CPPUNIT_ASSERT( edge == sameedge );
  CPPUNIT_ASSERT( otheredge == otheredge );
  CPPUNIT_ASSERT( anotheredge == anotheredge );
  CPPUNIT_ASSERT( yetanotheredge == yetanotheredge );

  CPPUNIT_ASSERT( edge != otheredge );
  CPPUNIT_ASSERT( edge != anotheredge );
  CPPUNIT_ASSERT( edge != yetanotheredge );
  CPPUNIT_ASSERT( otheredge != anotheredge );
  CPPUNIT_ASSERT( otheredge != yetanotheredge );
  CPPUNIT_ASSERT( anotheredge != yetanotheredge );
}

/** UnitTest for operator=()
 */
void FragmentEdgeTest::assignmentTest()
{
  FragmentEdge edge(1,1);
  FragmentEdge sameedge(1,1);
  FragmentEdge otheredge(1,4);

  // check initial status
  CPPUNIT_ASSERT( edge == sameedge);
  CPPUNIT_ASSERT( edge != otheredge);

  // now assign
  sameedge = otheredge;
  otheredge = edge;

  // and check again
  CPPUNIT_ASSERT( edge != sameedge);
  CPPUNIT_ASSERT( edge == otheredge);

  // also self-assign
  edge = edge;
  CPPUNIT_ASSERT( edge == edge);
}


/** UnitTest for serialization
 */
void FragmentEdgeTest::serializeTest()
{
  FragmentEdge edge(1,1);
  // serialize
  std::stringstream outputstream;
  boost::archive::text_oarchive oa(outputstream);
  oa << edge;

  // deserialize
  FragmentEdge sameedge;
  std::stringstream returnstream(outputstream.str());
  boost::archive::text_iarchive ia(returnstream);
  ia >> sameedge;

  CPPUNIT_ASSERT( edge == sameedge );
}
