/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * FragmentEdge.cpp
 *
 *  Created on: Sep 24, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "FragmentEdge.hpp"

#include <iostream>

#include "CodePatterns/Assert.hpp"

FragmentEdge::FragmentEdge() :
  AtomicNumberPair(std::make_pair(0,0))
{}

FragmentEdge::FragmentEdge(const AtomicNumberPair_t &_pair) :
  AtomicNumberPair(_pair)
{
  ASSERT( AtomicNumberPair.first < AtomicNumberPair.second,
      "FragmentEdge::FragmentEdge() - when giving as pair, first must be less than second.");
}

FragmentEdge::FragmentEdge(const size_t _left, const size_t _right) :
  AtomicNumberPair( _left < _right ? std::make_pair(_left,_right) : std::make_pair(_right,_left))
{}

/** Assignment operator for class FragmentEdge.
 *
 * We need this assignment to overcome the constness of the member. For STL
 * containers default cstors and usable assignment ops are required. This
 * const_cast is only way to allow for const member variables.
 *
 * @param edge other instance to use in assignment
 * @return this instance for concatenation
 */
FragmentEdge& FragmentEdge::operator=(const FragmentEdge &edge)
{
  // check self-assignment
  if (this != &edge) {
    const_cast<AtomicNumberPair_t &>(this->AtomicNumberPair) = edge.AtomicNumberPair;
  }
  return *this;
}

bool FragmentEdge::operator<(const FragmentEdge &edge) const
{
  if (AtomicNumberPair.first < edge.AtomicNumberPair.first) {
    return true;
  } else if (AtomicNumberPair.first > edge.AtomicNumberPair.first) {
    return false;
  } else {
    if (AtomicNumberPair.second < edge.AtomicNumberPair.second)
      return true;
    else
      return false;
  }
}

bool FragmentEdge::operator>(const FragmentEdge &edge) const
{
  if (AtomicNumberPair.first > edge.AtomicNumberPair.first) {
    return true;
  } else if (AtomicNumberPair.first < edge.AtomicNumberPair.first) {
    return false;
  } else {
    if (AtomicNumberPair.second > edge.AtomicNumberPair.second)
      return true;
    else
      return false;
  }
}

bool FragmentEdge::operator==(const FragmentEdge &edge) const
{
  if (AtomicNumberPair.first != edge.AtomicNumberPair.first) {
    return false;
  } else {
    return (AtomicNumberPair.second == edge.AtomicNumberPair.second);
  }
}

std::ostream& operator<<(std::ostream &out, const FragmentEdge &edge)
{
  out << edge.AtomicNumberPair.first << "<->" << edge.AtomicNumberPair.second;
  return out;
}


// we need to explicitly instantiate the serialization functions
BOOST_CLASS_EXPORT_IMPLEMENT(FragmentEdge)
